//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSETID_H)
#define CANDERA_ASSETID_H

#include <Candera/Environment.h>

#include <FeatStd/Util/BitKit.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLibEntities.h>

namespace Candera {
    
/** @addtogroup AssetLoaderBase
 *  @{
 */

    /**
     * @brief The asset library type enumeration.
     * 
     * Each asset repository contains a number of asset libraries, each one 
     *  collecting objects of similar types (e.g. scenes, bitmaps etc). AssetLib 
     *  specifies the base type of the objects in the library.
     */
    enum AssetLib {
        AnimationLib,                  ///< Library for AnimationPlayer objects
        AnimationGroupLib,             ///< Library for AnimationGroupPlayer objects
        BitmapLib,                     ///< Library for Bitmap objects
        CameraGroupLib,                ///< Library for CameraGroup objects
        ClearModeLib,                  ///< Library for shared ClearMode objects
        DisplayLib,                    ///< Library for Display objects
        FontResourceLib,               ///< Library for TextRendering::Font resources
        RawResourceLib,                ///< Library for raw resources
        ReferencedTemplateLib,         ///< Library for ReferenceTemplate objects
        RenderTargetLib,               ///< Library for GraphicDeviceUnit objects
        Scene2DLib,                    ///< Library for Scene2D objects
        SceneLib,                      ///< Library for Scene objects
        ShaderLib,                     ///< Library for Shader objects
        Composite2DLib,                ///< Library for CompositeGroup2D objects
        CompositeLib,                  ///< Library for CompositeGroup objects
        TextStyleLib,                  ///< Library for TextRendering::Style objects
        ThemeLib,                      ///< Library for Theme objects
        TransitionLib,                 ///< Library for Transition objects
        VertexBufferLib,               ///< Library for VertexBuffer objects
        ScriptLib,                     ///< Library for Script objects
        StateMachineLib,               ///< Library for StateMachineBehaviorData objects
        AppearanceCollectionLib,       ///< Library for shared MultiPassAppearance objects
        AppearanceLib,                 ///< Library for shared Appearance objects
        MaterialLib,                   ///< Library for shared Material objects
        RenderModeLib,                 ///< Library for shared RenderMode objects
        ShaderParamSetterLib,          ///< Library for shared ShaderParamSetter objects
        TextureLib,                    ///< Library for shared Texture objects
        LanguagePackLib,               ///< Library for LanguagePack objects
        c_libCount      // for internal use only!
    };

    /** @} */ // end of AssetLoaderBase

    namespace Internal {

        struct AssetId {
            UInt8 m_libraryType;
            UInt8 m_reserved;
            UInt16 m_itemType;
            UInt32 m_libraryId;
            UInt16 m_instanceId;
            UInt16 m_sclId;
            UInt32 m_nodeId;

            bool operator==(const AssetId& other) const;
            bool operator!=(const AssetId& other) const;
            bool IsValid() const;

            static const AssetId& InvalidId();
        };

        FEATSTD_LINT_NEXT_EXPRESSION(1923, "To be fixed in next release.")
#define AssetIdLogStr "[libraryType: %d; itemType: %d; libraryId: %d; instanceId: %d; sclId: %d; nodeId: %d]"
#define AssetIdLogArgs(assetId) assetId.m_libraryType, assetId.m_itemType, assetId.m_libraryId, assetId.m_instanceId, assetId.m_sclId, assetId.m_nodeId

        static const Id c_invalidAssetId = 0U;

        class AssetIdFunctions {
            public:
            static const UInt32 c_validityBit = 0U;
            static const UInt32 c_isLibraryIdBit = 1U;
            static const UInt32 c_isShareableIdBit = 2U;

            static const UInt32 c_nodeIdBitOffset = 2U;
            static const UInt32 c_nodeIdBitSize = 30U;

            static const UInt32 c_libraryIdBitOffset = 3U;
            static const UInt32 c_libraryIdBitSize = 25U;

            static const UInt32 c_sharedLibraryIdBitOffset = 9U;
            static const UInt32 c_sharedLibraryIdBitSize = 19U;

            static const UInt32 c_instanceIdBitOffset = 3U;
            static const UInt32 c_instanceIdBitSize = 6U;

            static const UInt32 c_sclIdBitOffset = 28U;
            static const UInt32 c_sclIdBitSize = 4U;

            static bool IsValid(Id id)
            {
                return FeatStd::Internal::BitKit::TestBit(id, c_validityBit);
            }

            static bool IsLibraryId(Id id)
            {
                return FeatStd::Internal::BitKit::TestBit(id, c_isLibraryIdBit);
            }

            static bool IsNodeId(Id id)
            {
                return !IsLibraryId(id);
            }

            static bool IsShareable(Id id)
            {
                return FeatStd::Internal::BitKit::TestBit(id, c_isShareableIdBit);
            }

            static UInt32 GetNodeId(Id id)
            {
                return FeatStd::Internal::BitKit::ExtractBits(id, c_nodeIdBitOffset, c_nodeIdBitSize);
            }

            static UInt32 GetLibraryId(Id id)
            {
                return FeatStd::Internal::BitKit::ExtractBits(id, c_libraryIdBitOffset, c_libraryIdBitSize);
            }

            static UInt16 GetSharedInstanceId(Id id)
            {
                return static_cast<UInt16>(FeatStd::Internal::BitKit::ExtractBits(id, c_instanceIdBitOffset, c_instanceIdBitSize));
            }

            static UInt32 GetSharedLibraryId(Id id)
            {
                return FeatStd::Internal::BitKit::ExtractBits(id, c_sharedLibraryIdBitOffset, c_sharedLibraryIdBitSize);
            }

            static UInt16 GetSclId(Id id)
            {
                return static_cast<UInt16>(FeatStd::Internal::BitKit::ExtractBits(id, c_sclIdBitOffset, c_sclIdBitSize));
            }

            static UInt8 GetAssetIdLibraryType(const UInt32* rawId) {
                return static_cast<UInt8>(rawId[3] >> 24);
            }

            static UInt16 GetAssetIdNodeType(const UInt32* rawId) {
                return static_cast<UInt16>(rawId[3]);
            }

            static UInt32 GetAssetIdLibraryItemIdentifier(const UInt32* rawId) {
                return rawId[2];
            }

            static UInt16 GetAssetIdLibraryItemInstaceIdentifier(const UInt32* rawId) {
                return static_cast<UInt16>(rawId[1] >> 16);
            }

            static UInt16 GetAssetIdSolutionIdentifier(const UInt32* rawId) {
                return static_cast<UInt16>(rawId[1]);
            }

            static UInt32 GetAssetIdNodeIdentifier(const UInt32* rawId) {
                return rawId[0];
            }

            static AssetId CreateId(AssetLib libType, AssetObjectType itemType, Id libraryId, Id nodeId);
            static Id GetNodeId(AssetId id);
            static Id GetLibraryId(AssetId id);

            static AssetId GetAssetId(const UInt32* rawId);
        };

        inline bool AssetId::IsValid() const
        {
            return *this != InvalidId();
        }

        inline const AssetId& AssetId::InvalidId()
        {
            static const AssetId id = {0, 0, 0, 0, 0, 0, 0};
            return id;
        }
    }
}

#endif // CANDERA_ASSETID_H
