//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetId.h"
#include <CanderaAssetLoader/AssetLoaderBase/AssetIdMacros.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    namespace Internal {

        FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

        bool AssetId::operator == (const AssetId& other) const
        {
            return (m_libraryType == other.m_libraryType) &&
                (m_libraryId == other.m_libraryId) &&
                (m_nodeId == other.m_nodeId) &&
                (m_sclId == other.m_sclId);
        }

        bool AssetId::operator != (const AssetId& other) const
        {
            return !(*this == other);
        }

        AssetId AssetIdFunctions::GetAssetId(const UInt32* rawId)
        {
            AssetId id = {
                GetAssetIdLibraryType(rawId),
                0,
                GetAssetIdNodeType(rawId),
                GetAssetIdLibraryItemIdentifier(rawId),
                GetAssetIdLibraryItemInstaceIdentifier(rawId),
                GetAssetIdSolutionIdentifier(rawId),
                GetAssetIdNodeIdentifier(rawId)
            };
            return id;
        }
        
        AssetId AssetIdFunctions::CreateId(AssetLib libType, AssetObjectType itemType, Id libraryId, Id nodeId)
        {
            if (!IsLibraryId(libraryId)) {
                FEATSTD_LOG_ERROR("Invalid library-ID: %u", libraryId);
                return AssetId::InvalidId();
            }

            if (!IsNodeId(nodeId)) {
                FEATSTD_LOG_ERROR("Invalid node-ID: %u", nodeId);
                return AssetId::InvalidId();
            }

            bool isShareableLibrary = IsShareable(libraryId);
            FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: only getter, no side effect")
            AssetId assetId = {
                static_cast<UInt8>(libType),
                0,
                static_cast<UInt16>(itemType),
                isShareableLibrary ? GetSharedLibraryId(libraryId) : GetLibraryId(libraryId),
                isShareableLibrary ? GetSharedInstanceId(libraryId) : static_cast<UInt16>(0),
                GetSclId(libraryId),
                GetNodeId(nodeId)
            };
            return assetId;
        }

        Id AssetIdFunctions::GetNodeId(AssetId id)
        {
            if (id.IsValid()) {
                return CDA_NODE_ASSETID(id.m_nodeId);
            }
            else {
                return 0;
            }
        }

        Id AssetIdFunctions::GetLibraryId(AssetId id)
        {
            Candera::Id result = CDA_LIBRARY_ASSETID(id.m_sclId, id.m_libraryId);

            if (id.IsValid()) {
                switch (id.m_libraryType) {
                    FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 3008 Required Rule 6-4-5: All paths end in return")
                case AnimationLib:                  ///< Library for AnimationPlayer objects
                case AnimationGroupLib:             ///< Library for AnimationGroupPlayer objects
                case BitmapLib:                     ///< Library for Bitmap objects
                case CameraGroupLib:                ///< Library for CameraGroup objects
                case ClearModeLib:                  ///< Library for shared ClearMode objects
                case DisplayLib:                    ///< Library for Display objects
                case FontResourceLib:               ///< Library for TextRendering::Font resources
                case RawResourceLib:                ///< Library for raw resources
                case ReferencedTemplateLib:         ///< Library for ReferenceTemplate objects
                case RenderTargetLib:               ///< Library for GraphicDeviceUnit objects
                case Scene2DLib:                    ///< Library for Scene2D objects
                case SceneLib:                      ///< Library for Scene objects
                case ShaderLib:                     ///< Library for Shader objects
                case Composite2DLib:                ///< Library for CompositeGroup2D objects
                case CompositeLib:                  ///< Library for CompositeGroup objects
                case TextStyleLib:                  ///< Library for TextRendering::Style objects
                case ThemeLib:                      ///< Library for Theme objects
                case TransitionLib:                 ///< Library for Transition objects
                case VertexBufferLib:               ///< Library for VertexBuffer objects
                case ScriptLib:                     ///< Library for Script objects

                case LanguagePackLib:               ///< Library for LanguagePack objects
                    result = CDA_LIBRARY_ASSETID(id.m_sclId, id.m_libraryId);
                    break;

                case AppearanceCollectionLib:       ///< Library for shared MultiPassAppearance objects
                case AppearanceLib:                 ///< Library for shared Appearance objects
                case MaterialLib:                   ///< Library for shared Material objects
                case RenderModeLib:                 ///< Library for shared RenderMode objects
                case ShaderParamSetterLib:          ///< Library for shared ShaderParamSetter objects
                case TextureLib:                    ///< Library for shared Texture objects
                    result = CDA_SHARED_LIBRARY_ASSETID(id.m_sclId, id.m_libraryId, id.m_instanceId);
                    break;
                default:
                    break;
                }

            }
            return result;
        }

    }
}   // namespace Candera

