//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetGroup.h"

#include <MemoryPlatform.h>

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Math.h>

#include <CanderaAssetLoader/AssetLoaderBase/AssetConfig.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetRepository.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetSet.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetValidation.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultResourceProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/RenderModeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AssetHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ItemHeaderCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LibraryHeaderCffReader.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    namespace Internal {

        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        AssetGroup::AssetGroup():
            m_assetSetList(),
            m_assetData(),
            m_validationFlags(0),
            FEATSTD_SUPPRESS_MSC_WARNING_FOR_NEXT_EXPRESSION(4355, "'this' is just stored, but not used now.")
            m_assetConfigListener(*this)
        {
        }

        AssetGroup::~AssetGroup()
        {
            m_assetSetList.Clear();
        }

        AssetGroup::SharedPointer AssetGroup::Create()
        {
            return SharedPointer(ASSETLOADER_TRANSIENT_NEW(AssetGroup)());
        }

        bool AssetGroup::Initialize(AssetConfig& assetConfig, UInt32 validationFlags)
        {
            m_validationFlags = validationFlags;
            if (!assetConfig.AddAssetConfigListener(&m_assetConfigListener)) {
                FEATSTD_LOG_WARN("Listener could not be added to AssetConfig. Further added asset partitions might not be recognized.");
            }
            m_assetConfigListener.SetContainer(&assetConfig);

            for (UInt16 repositoryIndex = 0; repositoryIndex < assetConfig.GetAssetRepositoryCount(); ++repositoryIndex) {
                const AssetSet::SharedPointer& assetSet = AddAssetRepository(assetConfig.GetAssetRepository(repositoryIndex));
                if (assetSet.PointsToNull()) {
                    FEATSTD_LOG_ERROR("Failed to load AssetRepository.");
                    return false;
                }
            }

            if (m_assetSetList.IsEmpty()) {
                FEATSTD_LOG_ERROR("No AssetRepository found in AssetConfig.");
                return false;
            }

            const AssetSet::SharedPointer& firstAssetSet = *m_assetSetList.Begin();

            AssetDataHandle headerDataHandle(GetAssetHeaderHandle());
            if (!firstAssetSet->InitializeAssetVersionInfo(headerDataHandle, m_assetData.m_versionInfo)) {
                FEATSTD_LOG_ERROR("InitializeAssetVersionInfo failed.");
                return false;
            }

            m_assetData.m_simulationShaders = CFFReader::GetAssetSimulationShaders(headerDataHandle);
            m_assetData.m_platformName = firstAssetSet->GetName(CFFReader::GetAssetPlatformName(headerDataHandle));
            m_assetData.m_platformInstanceName = firstAssetSet->GetName(CFFReader::GetAssetPlatformInstanceName(headerDataHandle));
            m_assetData.m_fractionalNumberRepresentation = firstAssetSet->GetName(CFFReader::GetAssetFractionalNumberRepresentation(headerDataHandle));
            m_assetData.m_defaultCulture = firstAssetSet->GetName(CFFReader::GetAssetDefaultCultureName(headerDataHandle));
            m_assetData.m_masterThemeName = firstAssetSet->GetName(CFFReader::GetAssetMasterThemeName(headerDataHandle));
            if (!AssetValidation::ValidateAssetPlatform(m_assetData, validationFlags)) {
                FEATSTD_LOG_ERROR("Failed to create AssetSet.");
                return false;
            }

            return true;
        }

        AssetDataHandle AssetGroup::GetItemDataHandle(const AssetId& id, UInt16* repositoryId)
        {
            for (AssetSetList::Iterator it = m_assetSetList.Begin(); it != m_assetSetList.End(); ++it) {
                const ResourceDataHandle& libHandle = (*it)->CreateAssetItemDataHandle(id);
                if (libHandle.m_size != 0) {
                    if (repositoryId != 0) {
                        *repositoryId = libHandle.m_accessor.m_asset.m_repositoryId;
                    }
                    return CFFReader::GetItemDataHandle(AssetDataHandle(libHandle));
                }
            }

            if (id.m_itemType != 0) {
                //workaround for AnimationPlayer(0), it could be that actually an AnimationGroupPlayer(1) is searched
                FEATSTD_LOG_WARN("No AssetSet could find item information for library item with AssetId: "
                    AssetIdLogStr, AssetIdLogArgs(id));
            }
            return AssetDataHandle();
        }

        ResourceDataHandle AssetGroup::GetItemHeaderHandle(const AssetId& id)
        {
            for (AssetSetList::Iterator it = m_assetSetList.Begin(); it != m_assetSetList.End(); ++it) {
                const ResourceDataHandle& libHandle = (*it)->CreateAssetItemHeaderHandle(id);
                if (libHandle.m_size != 0) {
                    return libHandle;
                }
            }

            FEATSTD_LOG_WARN("No AssetSet could find header information for library item with AssetId: "
                AssetIdLogStr, AssetIdLogArgs(id));
            return ResourceDataHandle::InvalidHandle();
        }

        void AssetGroup::OnAssetRepositoryAdded(AssetRepository* assetRepository)
        {
            const AssetSet::SharedPointer& assetSet = AddAssetRepository(assetRepository);
            if (assetSet == 0) {
                FEATSTD_LOG_ERROR("Failed to load AssetRepository.");
                return;
            }
        }

        void AssetGroup::OnAssetRepositoryRemoved(const AssetRepository* assetRepository)
        {
            if (assetRepository != 0) {
                for (AssetSetList::Iterator it = m_assetSetList.Begin(); it != m_assetSetList.End(); ++it) {
                    AssetSet::SharedPointer assetSet = *it;
                    if (assetSet->IsWrapping(assetRepository)) {
                        static_cast<void>(m_assetSetList.Remove(assetSet));
                        break;
                    }
                }
            }
        }

        AssetSet::SharedPointer AssetGroup::AddAssetRepository(AssetRepository* assetRepository)
        {
            if (assetRepository == 0) {
                FEATSTD_LOG_ERROR("Null AssetRepository.");
                return AssetSet::SharedPointer(0);
            }

            const AssetSet::SharedPointer& assetSet = AssetSet::Create(*assetRepository);
            if (assetSet.PointsToNull()) {
                FEATSTD_LOG_ERROR("Failed to create AssetSet.");
                return AssetSet::SharedPointer(0);
            }

            // Add to reservedIds all the ids assigned to already loaded and initialized asset sets so that
            // the id of the asset set being loaded can be assigned a unique value
            // Reason for not storing the vector in asset group: asset group has long life duration, re-initialization has small overhead,
            // keeping vector in sync with the list is prevented
            // Reason for choosing Vector over Map: asset set count is usually small (less than 10) and for basic scenarios
            // there is no significant performance difference (=> prefer less memory allocations)
            FeatStd::Internal::Vector<UInt16> reservedIds;
            if (!reservedIds.Reserve(m_assetSetList.GetSize())) {
                FEATSTD_LOG_ERROR("Failed to create list of reserved asset set ids.");
                return AssetSet::SharedPointer(0);
            }
            for (AssetSetList::Iterator it = m_assetSetList.Begin(); it != m_assetSetList.End(); ++it) {
                if (!reservedIds.Add((*it)->GetId())) {
                    FEATSTD_LOG_ERROR("Failed to create list of reserved asset set ids.");
                    return AssetSet::SharedPointer(0);
                }
            }

            if (!m_assetSetList.Append(assetSet)) {
                FEATSTD_LOG_ERROR("Failed to add AssetSet to internal list.");
                return AssetSet::SharedPointer(0);
            }

            if (!assetSet->Initialize(m_validationFlags, reservedIds)) {
                static_cast<void>(m_assetSetList.Remove(assetSet));
                FEATSTD_LOG_ERROR("Asset set could not be initialized.");
                return AssetSet::SharedPointer(0);
            }

            return assetSet;
        }

        ResourceDataHandle AssetGroup::GetHeaderDataHandle(UInt32(*fn)(const AssetDataHandle&), Int32 size) const
        {
            AssetDataHandle headerDataHandle(GetAssetHeaderHandle());
            if (!headerDataHandle.IsValid()) {
                return ResourceDataHandle::InvalidHandle();
            }

            return (*m_assetSetList.Begin())->CreateAssetHandle(fn(headerDataHandle), size);
        }

        AssetGroup::AssetSetIterator AssetGroup::GetAssetSetIterator()
        {
             return AssetSetIterator(m_assetSetList.Begin(), m_assetSetList.End());
        }

        bool AssetGroup::InitializeAssetVersionInfo(AssetData::AssetVersionInfo& versionInfo) const
        {
            return (*m_assetSetList.Begin())->InitializeAssetVersionInfo(AssetDataHandle(GetAssetHeaderHandle()), versionInfo);
        }

        ResourceDataHandle AssetGroup::GetAssetHeaderHandle() const
        {
            return (*m_assetSetList.Begin())->CreateAssetHandle(0, CFFReader::CFF_ASSET_HEADER_SIZE);
        }

        AssetDataHandle AssetGroup::GetLibItemHeader(const AssetId& id)
        {
            AssetDataHandle result;

            if (id.m_nodeId == 0) {
                return result;
            }

            AssetDataHandle itemHeader(GetItemHeaderHandle(id));
            if (!itemHeader.IsValid()) {
                return result;
            }

            const AssetDataHandle& itemListHandle = CFFReader::GetItemListHandle(itemHeader);
            if (!itemListHandle.IsValid()) {
                return result;
            }

            Int32 leftIndex = 0;
            Int32 rightIndex = CFFReader::GetItemCount(itemListHandle) - 1;
            while (leftIndex <= rightIndex) {
                Int32 middleIndex = (leftIndex + rightIndex) / 2;
                const AssetDataHandle& itemHandle = CFFReader::GetItem(itemListHandle, middleIndex);
                UInt32 nodeIdentifier = AssetIdFunctions::GetAssetIdNodeIdentifier(CFFReader::GetLibElementId(itemHandle));
                if (id.m_nodeId < nodeIdentifier) {
                    rightIndex = middleIndex - 1;
                }
                else if (id.m_nodeId > nodeIdentifier) {
                    leftIndex = middleIndex + 1;
                }
                else {
                    result = itemHandle;
                    break;
                }
            }

            return result;
        }

        AssetSet* AssetGroup::GetAssetSet(UInt16 id)
        {
            for (AssetSetIterator it = GetAssetSetIterator(); it.IsValid(); ++it) {
                if ((*it)->GetId() == id) {
                    return it->GetPointerToSharedInstance();
                }
            }

            FEATSTD_LOG_ERROR("AssetSet with Id %d not found", id);
            return 0;
        }

        AssetGroup::AssetGroupAssetConfigListener::AssetGroupAssetConfigListener(AssetGroup& assetGroup):
            m_assetGroup(assetGroup),
            m_container(0)
        {
        }

        AssetGroup::AssetGroupAssetConfigListener::~AssetGroupAssetConfigListener()
        {
            if (m_container != 0) {
                if (!m_container->RemoveAssetConfigListener(this)) {
                    FEATSTD_LOG_WARN("Listener about to be destroyed could not be removed from container.");
                }
                m_container = 0;
            }
        }

        void AssetGroup::AssetGroupAssetConfigListener::OnAssetRepositoryAdded(AssetRepository* assetRepository)
        {
            m_assetGroup.OnAssetRepositoryAdded(assetRepository);
        }

        void AssetGroup::AssetGroupAssetConfigListener::OnAssetRepositoryRemoved(AssetRepository* assetRepository)
        {
            m_assetGroup.OnAssetRepositoryRemoved(assetRepository);
        }

        void AssetGroup::AssetGroupAssetConfigListener::OnAssetConfigDestroy(AssetConfig* assetConfig)
        {
            if (assetConfig == m_container) {
                m_container = 0;
            }
        }

        ResourceDataHandle AssetGroup::GetDefaultRenderModeHandle() const
        {
            return GetHeaderDataHandle(&CFFReader::GetAssetDefaultRederModeOffset, CFFReader::CFF_RENDER_MODE_SIZE);
        }

        ResourceDataHandle AssetGroup::GetAutoUniformsSemanticsHandle() const
        {
            return GetHeaderDataHandle(&CFFReader::GetAssetAutoUniformsOffset, -1);
        }

        ResourceDataHandle AssetGroup::GetShaderAttributesSemanticsHandle() const
        {
            return GetHeaderDataHandle(&CFFReader::GetAssetShaderAtributesOffset, -1);
        }

        ResourceDataHandle AssetGroup::GetDevicePackageMetaInfoHandle() const
        {
            return GetHeaderDataHandle(&CFFReader::GetAssetDevicePackageOffset, -1);
        }

        ResourceDataHandle AssetGroup::GetTransitionRuleCollectionHandle() const
        {
            return GetHeaderDataHandle(&CFFReader::GetTransitionRuleCollectionOffset, -1);
        }


    }   // namespace Internal
}   // namespace Candera
