//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_AssetFontStore_h)
#define Candera_AssetFontStore_h

#include <Candera/TextEngine/FontStore.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Container/Map.h>
#include <FeatStd/Platform/String.h>

namespace Candera {
/** @addtogroup AssetLoaderBase
 *  @{
 */

    /** @brief implements a font store for the asset library
     *  this is the default font store Candera will use to retrieve font data from.
     *  This is a Candera internal class and must not be used by other components
     */
    class AssetFontStore : public TextRendering::FontStore {
        public:
            /**
            * Constructor
            */
            AssetFontStore() : m_loadStrategySelector(0) {};

            /**
             * Destructor
             */
            virtual ~AssetFontStore();

            /**
             * Retrieves descriptor ID and height of the default font.
             * @param descId Descriptor Id of default font, out parameter.
             * @param height height of default font, out parameter.
             * @see FontStore::DefaultFont
             */
            virtual void DefaultFont(UInt8 &descId, Int16 &height) const override;

            /**
             * Return the descriptor id for a given font name
             * @param name name of the font face (e.g. OpenSans). This string can be released as soon
             *             as the function returns.
             * @param faceIndex
             * @return descriptor id of the font, InvalidDescriptorId if the font does not exist in the store
             * @see FontStore::GetDescriptorId
             */
            virtual UInt8 GetDescriptorId(const Char *name, UInt8 faceIndex) override;

            /**
             * Load the font data into the store
             * loading must not necessarily mean loading from file, but preparing
             * what ever is necessary to read the data according to the FontResourceDescriptor
             * specification.
             * @param descId descriptor id of the font ( @see GetDescriptorId )
             * @param descriptor receives the information how to access the font data
             * @return true if the loading succeeded, false otherwise
             * @see FontStore::Load
             */
            virtual bool Load(UInt8 descId, TextRendering::FontResourceDescriptor &descriptor) override;

            /**
             * Unload the font data from the store.
             * The descriptor id does not become invalid. A new call to GetDescriptorId
             * is not required.
             * @param descId descriptor id of the font to unload.
             * @return true if unloading succeeded, false otherwise
             * @see FontStore::Unload
             */
            virtual bool Unload(UInt8 descId) override;

            /**
            * Retrieves the face name.
            * @param descId Description Id of face.
            * @return The face name.
            * @see FontStore::GetFaceName
            */
            virtual const Char* GetFaceName(UInt8 descId) override;

            /**
            * Retrieves the face index.
            * @param descId Description Id of face.
            * @return The face index.
            * @see FontStore::GetFaceIndex
            */
            virtual UInt8 GetFaceIndex(UInt8 descId) override;

            /**
            *  Returns the shared font buffer for a given face.
            *  @param name name of the font face (e.g. OpenSans)
            *  @param faceIndex index of the face in collection(e.g. index of the Italic face). Should be 0 for fonts that contain only one face.
            *  @return the shared font buffer.
            */
            virtual TextRendering::FontBuffer::SharedPointer GetFontBuffer(const Char *name, UInt8 faceIndex) override;

            /**
            *  Releases all stored shared font buffer objects which are solely referenced by the font store.
            *
            */
            virtual void ReleaseOrphanedFontBuffer() override;

            /**
            * Base class for the LoadStrategy selector.
            */
            class LoadStrategySelector {
                friend class AssetFontStore;
            public:
                /**
                * Select the font load strategy for AssetFontStore
                */
                enum LoadStrategy {
                    PreloadStrategy,        ///<Load the entire font data into memory if it is not already in a direct addressable asset repository. 
                                            /// Freetype will use the font in memory for text rendering/shaping.
                    OnRequestLoadStrategy ///<Do not load anything in memory, but read/copy data into smaller temporary buffers as requested by Freetype 
                                          /// during text rendering/shaping.
                };
            private:
                /**
                * Retrieve the strategy to use for loading fonts.
                * @param name Name of the font to load.
                * @return load strategy.
                */
                virtual LoadStrategy GetLoadStrategy(const Char* name) const = 0;
            };

            /**
            * Set a Font load strategy selector.
            * @param selector Font load strategy selector.
            */
            void SetLoadStrategySelector(LoadStrategySelector* selector) { m_loadStrategySelector = selector; }

            /**
            * Retrieve the Font load strategy selector.
            * @return Font load strategy selector.
            */
            LoadStrategySelector* GetLoadStrategySelector() const { return m_loadStrategySelector; }
        private:
            struct AssetFontMapEntry {
                Char *name;
                UInt faceIndex : 8;
                UInt isLoaded : 1;
                LoadStrategySelector::LoadStrategy m_loadStrategy;
            };
            typedef Internal::Map<const Char*, TextRendering::FontBuffer::SharedPointer> BufferMap;

            TextRendering::FontBuffer::SharedPointer GetFontBufferInternal(UInt8 descId);
            UInt8 IndexToDescriptor(Int index) const;
            Int DescriptorToIndex(UInt8 descId) const;
            static UInt32 StreamReadFunction(void* buffer, UInt32 offset, UInt32 count, const void* data);

            Candera::Internal::Vector<AssetFontMapEntry> m_store;
            BufferMap m_bufferMap;
            LoadStrategySelector* m_loadStrategySelector;
    };

    /**
    * LoadStrategySelector that has a default load strategy and an exception list from the default strategy.
    */
    class DefaultLoadStrategySelector : public AssetFontStore::LoadStrategySelector {
    public:
        DefaultLoadStrategySelector() :m_defaultLoadStrategy(PreloadStrategy), m_defaultFontLoadStrategyExceptionList(0) { }

        /**
        * @param strategy The default strategy will be used for all fonts that are not in the exception list.
        */
        void SetDefaultFontLoadStrategy(LoadStrategy strategy) { m_defaultLoadStrategy = strategy; }

        /**
        * @return The default strategy that will be used for all fonts that are not in the exception list.
        */
        LoadStrategy GetDefaultFontLoadStrategy() const  { return m_defaultLoadStrategy; }

        /**
        * Set a zero terminated list of char arrays that defines the fonts that will not use the default load strategy.
        * Example:
        * @code
                selector.SetDefaultFontLoadStrategy(Candera::AssetFontStore::LoadStrategySelector::OnRequestLoadStrategy);
                // "Bitstream Vera Sans" font shall be loaded with PreloadStrategy:
                const Char* g_fontName[2] = { "ConstructionKit##ConstructionKit#Resources#Fonts#Bitstream Vera Sans", 0 };
                selector.SetDefaultFontLoadStrategyExceptionList(g_fontName);
                fontStore.SetLoadStrategySelector(&selector);
        * @endcode
        * @param exceptionList A zero terminated list of char arrays that defines the fonts that will not use the default load strategy.
        */
        void SetDefaultFontLoadStrategyExceptionList(const Char** exceptionList) { m_defaultFontLoadStrategyExceptionList = exceptionList; }

        /**
        * @return The zero terminated list of char arrays that defines the fonts that will not use the default load strategy.
        */
        const Char** GetDefaultFontLoadStrategyExceptionList() const  { return m_defaultFontLoadStrategyExceptionList; }

    private:
        virtual LoadStrategy GetLoadStrategy(const Char* name) const;

        LoadStrategy m_defaultLoadStrategy;
        const Char** m_defaultFontLoadStrategyExceptionList;
    };

 /** @} */ // end of AssetLoaderBase
}    // namespace Candera
#endif    // Candera_AssetFontStore_h

