//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetFontStore.h"

#include <Candera/EngineBase/Common/ResourceObject.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <Candera/TextEngine/Types.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/ContentLoader.h>
#include <Candera/EngineBase/Common/ResourceProvider.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <FeatStd/Util/NumericUtil.h>
#include <FeatStd/Util/PointerUtil.h>


namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

    class FontDataDisposer : public Candera::MemoryManagement::DisposerBase<const void*> {
    public:
        static void Dispose(const void* data)
        {
            ASSETLOADER_DELETE_ARRAY(data);
        }
    };

    // -------------------------------------------------------------------------
    AssetFontStore::~AssetFontStore()
    {
        SizeType size = m_store.Size();

        for (SizeType i = 0; i < size; i++) {
            if (m_store[i].name != 0) {
                ASSETLOADER_DELETE_ARRAY(m_store[i].name);
                m_store[i].name = 0;
            }
        }
        m_bufferMap.Clear();
        m_store.Clear();
        m_loadStrategySelector = 0;
    }

    // -------------------------------------------------------------------------
    void AssetFontStore::DefaultFont(UInt8 &descId, Int16 &height) const
    {
        //TODO: this is not good... check how to retrieve default values
        descId = InvalidDescriptorId;
        height = 0;
    }

    // -------------------------------------------------------------------------
    UInt8 AssetFontStore::GetDescriptorId(const Char *name, UInt8 faceIndex)
    {
        if (name == 0) {
            return InvalidDescriptorId;
        }
        Int index = -1;
        Int unused = -1;
        Int size = static_cast<Int>(m_store.Size());
        Char* existingName = 0;

        for (Int i = 0; i < size; i++) {
            if (m_store[i].name == 0) {
                if (unused < 0) {
                    //remember the first unused item
                    unused = i;
                }
            }
            else if (StringPlatform::CompareStrings(name, m_store[i].name) == 0) {
                if (existingName == 0) {
                    existingName = m_store[i].name;
                }
                if (m_store[i].faceIndex == faceIndex) {
                    //stop if an entry matches
                    index = i;
                    break;
                }
            }
            else {
                //do nothing
            }
        }

        if (index < 0) {
            //if item was not found initialize an item
            if (unused < 0){
                //if there are no unused entries create one
                AssetFontMapEntry item = { 0, 0, false, LoadStrategySelector::PreloadStrategy};

                if (m_store.Add(item)){
                    unused = size;
                    // A new entry means that we cannot reuse the existing string, otherwise
                    // the destructor will later attempt to free the same string twice causing a crash.
                    existingName = 0;
                }
            }
            if (unused >= 0) {
                //if there is an unused item
                if (existingName == 0) {
                    SizeType len = StringPlatform::Length(name);

                    Char* newName = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, len + 1);
                    if (newName != 0) {
                        StringPlatform::Copy(newName, name);
                    }
                    existingName = newName;
                }

                AssetFontMapEntry entry = {existingName, faceIndex, false, LoadStrategySelector::PreloadStrategy};
                m_store[unused] = entry;
                if (m_loadStrategySelector != 0)
                {
                    m_store[unused].m_loadStrategy = m_loadStrategySelector->GetLoadStrategy(name);
                }

                index = unused;
            }
        }

        return IndexToDescriptor(index);
    }

    // -------------------------------------------------------------------------
    namespace {
        TextRendering::FontBuffer::SharedPointer& GetDefaultFontBuffer()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(TextRendering::FontBuffer::SharedPointer, fontBuffer);
            return fontBuffer;
        }
    }

    // -------------------------------------------------------------------------
    TextRendering::FontBuffer::SharedPointer AssetFontStore::GetFontBuffer(const Char *name, UInt8 faceIndex)
    {
        UInt8 descriptor = GetDescriptorId(name, faceIndex);
        return GetFontBufferInternal(descriptor);
    }

    // -------------------------------------------------------------------------
    TextRendering::FontBuffer::SharedPointer AssetFontStore::GetFontBufferInternal(UInt8 descId)
    {
        Int index = DescriptorToIndex(descId);
        bool success = index >= 0;
        success = success && (ContentLoader::GetInstance().GetAssetProvider() != 0);

        if (success)
        {
            AssetFontMapEntry& entry = m_store[index];
            TextRendering::FontBuffer::SharedPointer* fontBuffer = m_bufferMap.Find(entry.name);
            if (0 == fontBuffer)
            {
                TextRendering::FontBuffer::SharedPointer newBuffer = TextRendering::FontBuffer::Create();
                if (!m_bufferMap.Insert(entry.name, newBuffer)) {
                    return GetDefaultFontBuffer();
                }
                fontBuffer = m_bufferMap.Find(entry.name);
            }
            return *fontBuffer;
        }

        return GetDefaultFontBuffer();
    }

    // -------------------------------------------------------------------------
    bool AssetFontStore::Load(UInt8 descId, Candera::TextRendering::FontResourceDescriptor &descriptor)
    {
        Int index = DescriptorToIndex(descId);
        if (index < 0){
            return false;
        }

        if (ContentLoader::GetInstance().GetAssetProvider() == 0) {
            return false;
        }

        bool success = true;
        AssetFontMapEntry& entry = m_store[index];
        AssetProvider &assetProvider = *ContentLoader::GetInstance().GetAssetProvider();
        Candera::Internal::AssetId result = assetProvider.GetAssetId(FontResourceLib, entry.name);
        if (!result.IsValid()) {
            FEATSTD_LOG_DEBUG("GetAssetId is not valid");
        }
        TextRendering::FontBuffer::SharedPointer fontBuffer = GetFontBufferInternal(descId);
        ResourceDataHandle resHdl = assetProvider.GetFontResourceById(Internal::AssetIdFunctions::GetLibraryId(result));
        UInt32 size = resHdl.m_size;
        switch (entry.m_loadStrategy)
        {
            case LoadStrategySelector::PreloadStrategy:
            {
                const void* cdata = 0;

                if (entry.isLoaded == 0)
                {
                    cdata = Internal::ResourceData::GetAddress(resHdl);
                    
                    if (cdata == 0) {
                        void* data = ASSETLOADER_TRANSIENT_NEW_ARRAY(UInt8, size);
                        if (data != 0) {
                            if (size != Internal::ResourceData::CopyData(data, resHdl, 0, size)) {
                                ASSETLOADER_DELETE_ARRAY(data);
                                data = 0;
                            }
                            else
                            {
                                fontBuffer->SetData(data, FontDataDisposer::Dispose);
                            }
                        }
                        cdata = data;
                    }
                    else
                    {
                        fontBuffer->SetData(cdata, 0);
                    }

                    if (cdata != 0) {
                        fontBuffer->SetAllocated(true);
                        fontBuffer->SetSize(size);
                        entry.isLoaded = 1;
                    }
                }
                else {
                    if (entry.isLoaded == 0) {
                        entry.isLoaded = 1;
                    }

                    cdata = fontBuffer->GetData();
                    size = FeatStd::Internal::NumericConversion<UInt32, SizeType>(fontBuffer->GetSize());
                }

                if (cdata == 0) {
                    success = false;
                }
                else
                {
                    descriptor.faceIndex = entry.faceIndex;
                    descriptor.storageType = TextRendering::FontResourceDescriptor::MemoryResource;
                    descriptor.storageSpec.memoryLocation.baseAddress = cdata;
                    descriptor.storageSpec.memoryLocation.nBytes = size;
                    descriptor.uid = descId;
                }
                break;
            }
            case LoadStrategySelector::OnRequestLoadStrategy:
            {
                if (size > 0) {
                    descriptor.faceIndex = entry.faceIndex;
                    descriptor.storageType = TextRendering::FontResourceDescriptor::StreamResource;
                    descriptor.storageSpec.streamSpecification.size = size;
                    descriptor.storageSpec.streamSpecification.data = entry.name;
                    descriptor.storageSpec.streamSpecification.readFunction = &StreamReadFunction;
                    descriptor.uid = descId;
                }
                else {
                    success = false;
                }
                break;
            }
            default:
                success = false;
                break;
        }
        return success;
    }

    // -------------------------------------------------------------------------
    bool AssetFontStore::Unload(UInt8 descId)
    {
        Int index = DescriptorToIndex(descId);
        if (index < 0) {
            return false;
        }

        if (m_store[index].isLoaded == 1) {
            TextRendering::FontBuffer::SharedPointer* fontBuffer = m_bufferMap.Find(m_store[index].name);
            if (fontBuffer != 0) {
                // if the map contains the last reference to the font buffer, then remove it.
                // The SharedPointer's retain count will go to 0 and the font buffer be destroyed.
                // This will be the case when 2 conditions are fulfilled:
                // 1.) The SharedStyle keeping Font objects gets destroyed
                //     e.g. DefaultAssetProvider::ReleaseOrphanedObjects() has been called.
                // 2.) The Font Engine's cache management calls this method sometimes afterwards.
                // Remark: Between 1 and 2 a SharedStyle might be created with a Font requesting the same font buffer.
                //         In that case, the resource will be kept.
                if (FeatStd::MemoryManagement::Internal::IsSoleReference(*fontBuffer)) {
                    static_cast<void>(m_bufferMap.Remove(m_store[index].name));
                    m_store[index].isLoaded = 0;
                }
            }
        }

        return true;
    }

    // -------------------------------------------------------------------------
    void AssetFontStore::ReleaseOrphanedFontBuffer()
    {
        SizeType size = m_store.Size();
        for (SizeType i = 0; i < size; i++) {
            if (m_store[i].isLoaded == 1) {
                TextRendering::FontBuffer::SharedPointer* fontBuffer = m_bufferMap.Find(m_store[i].name);
                if (fontBuffer != 0) {
                    if (FeatStd::MemoryManagement::Internal::IsSoleReference(*fontBuffer)) {
                        static_cast<void>(m_bufferMap.Remove(m_store[i].name));
                        m_store[i].isLoaded = 0;
                    }
                }
            }
        }
    }

    // -------------------------------------------------------------------------
    UInt32 AssetFontStore::StreamReadFunction(void* buffer, UInt32 offset, UInt32 count, const void* data)
    {
        if (count == 0) {
            return 0;
        }
        if ((buffer == 0) || (data == 0)) {
            return 0;
        }

        AssetProvider &assetProvider = *ContentLoader::GetInstance().GetAssetProvider();
        ResourceDataHandle resHdl = assetProvider.GetFontResourceById(Internal::AssetIdFunctions::GetLibraryId(assetProvider.GetAssetId(FontResourceLib, FeatStd::Internal::PointerToPointer<const Char*>(data))));

        UInt32 fontSize = resHdl.m_size;
        if (fontSize > 0) {
            if (fontSize > offset) {
                const void* address = Internal::ResourceData::GetAddress(resHdl);

                if ((offset + count) > fontSize) {
                    count = fontSize - offset;
                }
                if (address == 0) {
                    count = FeatStd::Internal::NumericConversion<UInt32>(Internal::ResourceData::CopyData(buffer, resHdl, static_cast<OffsetType>(offset), static_cast<SizeType>(count)));
                }
                else {
                    MemoryPlatform::Copy(buffer, FeatStd::Internal::PointerAdd<const void*, const void>(address, offset), static_cast<SizeType>(count));
                }
                return count;
            }
        }
        return 0;
    }

    // -------------------------------------------------------------------------
    AssetFontStore::LoadStrategySelector::LoadStrategy DefaultLoadStrategySelector::GetLoadStrategy(const Char* name) const
    {
        if (m_defaultFontLoadStrategyExceptionList != 0) {
            for (Int32 index = 0; m_defaultFontLoadStrategyExceptionList[index] != 0; index++) {
                if (StringPlatform::CompareStrings(name, m_defaultFontLoadStrategyExceptionList[index]) == 0) {
                    if (m_defaultLoadStrategy == PreloadStrategy) {
                        return OnRequestLoadStrategy;
                    }
                    else {
                        return PreloadStrategy;
                    }
                }
            }
        }

        return m_defaultLoadStrategy;
    }

    // -------------------------------------------------------------------------
    const Char* AssetFontStore::GetFaceName(UInt8 descId)
    {
        Int index = DescriptorToIndex(descId);
        if (index >= 0) {
            return m_store[index].name;
        }
        else {
            return 0;
        }
    }

    // -------------------------------------------------------------------------
    UInt8 AssetFontStore::GetFaceIndex(UInt8 descId)
    {
        Int index = DescriptorToIndex(descId);
        if (index >= 0) {
            return m_store[index].faceIndex;
        }
        else {
            return 0;
        }
    }

    // -------------------------------------------------------------------------
    UInt8 AssetFontStore::IndexToDescriptor(Int index) const
    {
        if ((index < 0) || (static_cast<SizeType>(index) >= m_store.Size())) {
            return InvalidDescriptorId;
        }
        return static_cast<UInt8>(index + 1);
    }

    // -------------------------------------------------------------------------
    Int AssetFontStore::DescriptorToIndex(UInt8 descId) const
    {
        if (descId == InvalidDescriptorId) {
            return -1;
        }
        Int index = static_cast<Int>(descId - 1);
        if (static_cast<SizeType>(index) >= m_store.Size()){
            return -1;
        }
        return index;
    }
}    // namespace Candera
