//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ASSET_CONFIG_H)
    #define CANDERA_ASSET_CONFIG_H

#include <CanderaPlatform/OS/CanderaTypes.h>
#include <Candera/EngineBase/Common/ListenerContainer.h>

namespace Candera {

/** @addtogroup AssetLoaderBase
 *  @{
 */

    class AssetRepository;
    class AssetConfig;

    /**
    * @brief A AssetConfigListener defines hooks that are called when asset
    *  repositories are added or removed.
    */
    class AssetConfigListener
    {
    public:
        /**
        * Constructs a AssetConfigListener object.
        */
        AssetConfigListener() {}

        /**
        * Destructs a AssetConfigListener object.
        */
        virtual ~AssetConfigListener() {}

        /**
        * Is invoked after an AssetRepository is added.
        * @param assetRepository The added AssetRepository.
        */
        virtual void OnAssetRepositoryAdded(AssetRepository* assetRepository) = 0;

        /**
        * Is invoked after an AssetRepository is removed.
        * @param assetRepository The removed AssetRepository.
        */
        virtual void OnAssetRepositoryRemoved(AssetRepository* assetRepository) = 0;

        /**
         * Is invoked by the destructor of the AssetConfig.
         * Instance of the AssetConfig that is destroyed.
         */
        virtual void OnAssetConfigDestroy(AssetConfig* /*assetConfig*/) {}
    };

    /**
     * @brief Asset configuration that encapsulates various asset repositories
     * Custom AssetConfig classes can be derived from this interface that
     * can encapsulate multiple asset repositories, even of different types.
     * All asset repositories will be used by the AssetProvider.
     */
    class AssetConfig {

        public:
            /**
             *  Constructor
             */
            AssetConfig() {}

            /**
             *  Destructor
             */
            virtual ~AssetConfig();

            /**
             *  Retrieves the asset repository.
             *  @param index The index
             *  @return The asset repository.
             */
            virtual AssetRepository* GetAssetRepository(UInt16 index) const = 0;

            /**
             *  Retrieves the asset repository count.
             *  @return The asset repository count.
             */
            virtual UInt16 GetAssetRepositoryCount() const = 0;

            bool AddAssetConfigListener(AssetConfigListener* listener);
            bool RemoveAssetConfigListener(AssetConfigListener* listener);
        protected:
            /**
             *  Has to be called in derived classes whenever a repository has been added during runtime (after the
             *  asset provider has been initialized with the startup configuration).
             *  @param  assetRepository     Pointer to the added repository.
             */
            void OnAssetRepositoryAdded(AssetRepository* assetRepository);

            /**
             *  Has to be called in derived classes whenever a repository has been removed during runtime (after the
             *  asset provider has been initialized with the startup configuration).
             *  @param  assetRepository     Pointer to the removed repository.
             */
            void OnAssetRepositoryRemoved(AssetRepository* assetRepository);

        private:

            class OnAssetRepositoryAddedEvent;
            class OnAssetRepositoryRemovedEvent;
            class OnAssetConfigDestroyEvent;

            typedef Internal::ListenerContainer<AssetConfigListener> ListenerContainer;
            ListenerContainer m_listenerContainer;
    };

/** @} */ // end of AssetLoaderBase

}   // namespace Candera

#endif  // CANDERA_ASSET_CONFIG_H
