//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AssetCache.h"

#include <FeatStd/Util/StaticObject.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>

namespace Candera {
    static bool InsertMetaDataInAssetCache(Internal::AssetCache::LocationMap &map, Internal::AssetCache::Key key, Internal::AssetCache::MetaData* data)
    {
        return map.Insert(key, data);
    }
    namespace Internal {

        using namespace FeatStd::Internal;

        FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);
        
        static const UInt32 c_bufferSize = CANDERA_ASSET_CACHE_SIZE;

        static const AssetCache::Key& InvalidKey()
        {
            static const AssetCache::Key key = {0, 0, 0};
            return key;
        }

        AssetCache::AssetCache()
            :m_buffer(ASSETLOADER_PERMANENT_ALLOC(c_bufferSize)),
            m_nextOffset(0)
        {
            MemoryPlatform::Set(m_buffer, 0U, c_bufferSize);
            MetaData* firstBlock = PointerToPointer<MetaData*>(m_buffer);
            firstBlock->m_offsetToNextMetaData = c_bufferSize;
        }

        AssetCache::~AssetCache()
        {
            ASSETLOADER_FREE(m_buffer);
            m_buffer = 0;
        }
        AssetCache& AssetCache::GetInstance()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(AssetCache, cache);
            return cache;
        }

        void* AssetCache::GetAndRetain(const Key& key)
        {
            MetaData** metaData = m_locationMap.Find(key);
            if (metaData != 0) {
                ++(*metaData)->m_retainCount;
                return *metaData + 1;
            }

            return 0;
        }

        void AssetCache::Retain(const void* data) const
        {
            FEATSTD_DEBUG_ASSERT(data != 0);
            MetaData* metaData = const_cast<MetaData*>(PointerToPointer<const MetaData*>(data) - 1);
            ++metaData->m_retainCount;
        }

        void AssetCache::Release(const void* data)
        {
            FEATSTD_DEBUG_ASSERT(data != 0);
            MetaData* metaData = const_cast<MetaData*>(PointerToPointer<const MetaData*>(data) - 1);
            --metaData->m_retainCount;

            if ((metaData->m_offsetToNextMetaData == c_notInCache) && (metaData->m_retainCount == 0)) {
                if (m_locationMap.Remove(metaData->m_key)) {
                    ASSETLOADER_FREE(FeatStd::Internal::PointerToPointer<void*>(metaData));
                }
                else {
                    FEATSTD_LOG_WARN("Failed to release data from asset cache.");
                }
            }
        }

        void* AssetCache::Reserve(const Key& key, UInt32 size)
        {
            if (size == 0) {
                return 0;
            }
            UInt32 requiredSize = ((size + (c_MetaDataSize + 3)) / 4) * 4;
            UInt32 candidateOffset = m_nextOffset;
            UInt32 nextOffset = m_nextOffset;
            bool overflow = false;

            while ((nextOffset - candidateOffset) < requiredSize)
            {
                if (nextOffset == c_bufferSize) {
                    if (overflow) {
                        FEATSTD_LOG_INFO("Insufficient memory in the cache. Additional %d bytes have been temporary allocated.", requiredSize);
                        void *result = ASSETLOADER_TRANSIENT_ALLOC(requiredSize);
                        if (0 != result) {
                            MetaData* metaData = PointerToPointer<MetaData*>(result);
                            metaData->m_key = key;
                            metaData->m_offsetToNextMetaData = c_notInCache;
                            metaData->m_retainCount = 1;
                            if (!m_locationMap.Insert(key, metaData)) {
                                FEATSTD_LOG_ERROR("Failed to retain data in asset cache.");
                            }
                            return metaData + 1;
                        }
                        else {
                            return 0;
                        }
                    }
                    nextOffset = 0;
                    candidateOffset = 0;
                    overflow = true;
                }

                MetaData* nextNetaData = GetMetaData(nextOffset);
                FEATSTD_DEBUG_ASSERT(nextNetaData->m_offsetToNextMetaData <= c_bufferSize);
                nextOffset = nextNetaData->m_offsetToNextMetaData;
                if (nextNetaData->m_retainCount != 0) {
                    candidateOffset = nextOffset;
                }

                if (overflow && (candidateOffset >= m_nextOffset)) {
                    FEATSTD_LOG_INFO("Insufficient memory in the cache. Additional %d bytes have been temporary allocated.", requiredSize);
                    void *result = ASSETLOADER_TRANSIENT_ALLOC(requiredSize);
                    if (result != 0) {
                        MetaData* metaData = PointerToPointer<MetaData*>(result);
                        metaData->m_key = key;
                        metaData->m_offsetToNextMetaData = c_notInCache;
                        metaData->m_retainCount = 1;
                        if (!InsertMetaDataInAssetCache(m_locationMap, key, metaData)) {
                            FEATSTD_LOG_ERROR("Failed to retain data in asset cache.");
                        }
                        return metaData + 1;
                    }
                    return 0;
                }
            }

            m_nextOffset = candidateOffset + requiredSize;
            UInt32 initOffset = 0;
            if ((nextOffset - m_nextOffset) < c_MetaDataSize) {
                m_nextOffset = nextOffset;
            }
            else {
                initOffset = nextOffset;
                nextOffset = m_nextOffset;
            }

            MetaData* resultMetaData = GetMetaData(candidateOffset);
            MetaData* metaData = resultMetaData;
            for (UInt32 newOffset = candidateOffset; newOffset < nextOffset; newOffset = metaData->m_offsetToNextMetaData) {
                metaData = GetMetaData(newOffset);
                if ((metaData->m_key.m_repositoryId != 0) || (metaData->m_key.m_regionIndex != 0) || (metaData->m_key.m_offset != 0)) {
                    static_cast<void>(m_locationMap.Remove(metaData->m_key));
                }
            }

            resultMetaData->m_offsetToNextMetaData = nextOffset;
            resultMetaData->m_key = key;
            resultMetaData->m_retainCount = 1;

            if (initOffset != 0) {
                metaData = GetMetaData(m_nextOffset);
                metaData->m_key = InvalidKey();
                metaData->m_offsetToNextMetaData = initOffset;
                metaData->m_retainCount = 0;
            }

            if (!InsertMetaDataInAssetCache(m_locationMap, key, resultMetaData)) {
                FEATSTD_LOG_ERROR("Failed to retain data in asset cache.");
            }

            return resultMetaData + 1;
        }
    }   // namespace Internal

    bool operator<(const Internal::AssetCache::Key& left, const Internal::AssetCache::Key& right)
        {
            return (left.m_repositoryId < right.m_repositoryId) || ((left.m_repositoryId == right.m_repositoryId) && ((left.m_regionIndex < right.m_regionIndex) || ((left.m_regionIndex == right.m_regionIndex) && (left.m_offset < right.m_offset))));
        }

}   // namespace Candera
