//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ScriptComponentCollectionAssetReader.h"
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <CanderaScripting/LuaScriptSystem.h>
#include <CanderaScripting/ScriptSystem.h>
#include <CanderaScripting/Script.h>
#include <Candera/System/EntityComponentSystem/ComponentSystem.h>
#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/NodeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ScriptComponentCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ScriptComponentCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ScriptParamCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {

    using namespace Diagnostics;

    FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

    namespace Internal {

        struct ScriptSystemWrapper
        {
            ScriptSystemWrapper() : m_scriptSystem(0), m_isCreated(false)
            {
                m_scriptSystem = EntityComponentSystem::EntitySystem::Get<Scripting::ScriptSystem>();
                if (m_scriptSystem == 0) {
                    m_scriptSystem = EntityComponentSystem::EntitySystem::Create<Scripting::LuaScriptSystem>();
                    if (m_scriptSystem != 0) {
                        if (m_scriptSystem->Init()) {
                            m_isCreated = true;
                        }
                        else {
                            if (EntityComponentSystem::EntitySystem::Destroy<Scripting::ScriptSystem>()) {
                                m_scriptSystem = 0;
                            }
                        }
                    }
                }

                // Initiate construction of static InstanceID<Script>, so it's guaranteed to exist in the destructor.
                SharedPointer<Scripting::Script> script = Scripting::Script::Create(0);
            }

            ~ScriptSystemWrapper()
            {
                if (m_isCreated) {
                    if (EntityComponentSystem::EntitySystem::Destroy<Scripting::ScriptSystem>()) {
                        m_scriptSystem = 0;
                    }
                }
            }
            Scripting::ScriptSystem* m_scriptSystem;
            bool m_isCreated;
        };

        static ScriptSystemWrapper& GetScriptSystemWrapper()
        {
            // Using synced instead of unsynced, because unsynced static objects are not thread-safe
            // and need to be created pre-main via a static var. However we only want to create the
            // script system if scripts are actually used.
            FEATSTD_SYNCED_STATIC_OBJECT(ScriptSystemWrapper, scriptSystemWrapper);
            return scriptSystemWrapper;
        }

        bool ScriptComponentCollectionAssetReader::ReadScriptComponentCollectionData(CanderaObject& canderaObject, const CffLoaderContext& context)
        {
            const AssetDataHandle& collectionAssetHandle = CFFReader::GetNodeScriptComponentCollection(context.handle);
            if (!collectionAssetHandle.IsValid()) {
                //No scripts attached.
                return true;
            }

            Int componentCount = CFFReader::GetScriptComponentCollectionChildrenCount(collectionAssetHandle);
            if (componentCount <= 0) {
                return true;
            }

            Scripting::ScriptSystem* componentSystem = GetScriptSystemWrapper().m_scriptSystem;
            if (componentSystem == 0) {
                FEATSTD_LOG_WARN("No script system is available to register script components.");
                return false;
            }

            bool result = true;

            for (Int componentIndex = 0; componentIndex < componentCount; ++componentIndex) {
                const AssetDataHandle& componentAssetHandle = CFFReader::GetScriptComponentCollectionChildrenElementAt(collectionAssetHandle, componentIndex);
                if (!componentAssetHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Asset error, invalid ScriptComponent handle.");
                    break;
                }

                Scripting::ScriptSystem::Handle componentHandle = componentSystem->CreateComponent();
                if (componentHandle.IsNullHandle()) {
                    FEATSTD_LOG_ERROR("Invalid ScriptComponent handle.");
                    break;
                }
#ifdef CANDERA_SCRIPTING_ENABLED
                const AssetId& scriptAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetScriptComponentScript(componentAssetHandle));
                MemoryManagement::SharedPointer<Scripting::Script> resultScript = context.provider->GetScriptByAssetId(scriptAssetId);
                if (resultScript.PointsToNull()){
                    FEATSTD_LOG_DEBUG("GetScriptByAssetId returns a null pointer.");
                }
                if (!componentSystem->SetComponentScript(componentHandle, resultScript)) {
                    FEATSTD_LOG_WARN("Script error: %s", componentSystem->GetLastCompileErrorMessage());
                    componentSystem->ClearLastCompileErrorMessage();
                    result = false;
                }
#endif
                if (!componentSystem->SetComponentEnabled(componentHandle, CFFReader::GetScriptComponentIsEnabled(componentAssetHandle))) {
                    FEATSTD_LOG_WARN("Failed to enable script component");
                }
                if (!componentSystem->SetComponentPriority(componentHandle, CFFReader::GetScriptComponentPriority(componentAssetHandle))) {
                    FEATSTD_LOG_WARN("Failed to set priority to script component");
                }

                for (Int parameterIndex = 0; parameterIndex < CFFReader::GetScriptComponentParametersCount(componentAssetHandle); ++parameterIndex) {
                    const AssetDataHandle& parameterAssetHandle = CFFReader::GetScriptComponentParametersElementAt(componentAssetHandle, parameterIndex);
                    if (!parameterAssetHandle.IsValid()) {
                        FEATSTD_LOG_ERROR("Asset error, invalid ScriptComponent handle.");
                        break;
                    }

                    const Char* paramName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetScriptParamName(parameterAssetHandle));
                    Scripting::ScriptComponent::ParameterType paramType = static_cast<Scripting::ScriptComponent::ParameterType>(CFFReader::GetScriptParamType(parameterAssetHandle));
                    const void* paramValue = CFFReader::GetScriptParamValue(parameterAssetHandle);
                    bool isParamSet = false;
                    switch (paramType) {
                        case Scripting::ScriptComponent::Number: 
                            isParamSet = componentSystem->AddComponentParameterNumber(componentHandle, paramName, *(FeatStd::Internal::PointerToPointer<const Double*>(paramValue))); break;
                        case Scripting::ScriptComponent::Integer:
                            isParamSet = componentSystem->AddComponentParameterInteger(componentHandle, paramName, *(FeatStd::Internal::PointerToPointer<const Int64*>(paramValue))); break;
                        case Scripting::ScriptComponent::Boolean:
                            isParamSet = componentSystem->AddComponentParameterBoolean(componentHandle, paramName, *(FeatStd::Internal::PointerToPointer<const bool*>(paramValue))); break;
                        case Scripting::ScriptComponent::String:
                            isParamSet = componentSystem->AddComponentParameterString(componentHandle, paramName, FeatStd::Internal::PointerToPointer<const Char*>(paramValue) + 2); break;
                        default:
                            break;
                    }

                    if (!isParamSet) {
                        FEATSTD_LOG_WARN("Parameter %s is not set", paramName);
                    }
                }

                if (!componentSystem->AttachComponent(componentHandle, &canderaObject)) {
                    FEATSTD_LOG_WARN("Script error: %s", componentSystem->GetLastCompileErrorMessage());
                    componentSystem->ClearLastCompileErrorMessage();
                    result = false;
                }
            }

            return result;
        }
    }
}
