//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SceneContextBaseAssetReader.h"
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/SceneContextBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/SceneCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/SceneAnimationInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaWidget/WidgetBase/WidgetMetaInfo.h>
#include <CanderaWidget/WidgetBase/WidgetBase.h>

#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/AnimationBlendedPropertyListAssetReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/WidgetBaseAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/WidgetCollectionAssetBuilder.h>

namespace Candera {
    using namespace Animation;
    using namespace MemoryManagement;
    using namespace MetaInfo;

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

    namespace Internal {

        class SceneContextBaseWidgetCollectionBuilder: public WidgetCollectionAssetBuilderBase {
        public:
            SceneContextBaseWidgetCollectionBuilder(SceneContextBase& sceneContext):m_sceneContext(sceneContext) {}
        private:
            virtual bool AddWidget(WidgetBase* widget) override
            { 
                return m_sceneContext.AddWidget(widget);
            }
            virtual WidgetBase* GetFirstWidget() override
            {
                WidgetBase* widget = 0;
                static_cast<void>(m_sceneContext.GetFirstWidget(widget));
                return widget;
            }
            virtual WidgetBase* GetNextWidget() override
            {
                WidgetBase* widget = 0;
                static_cast<void>(m_sceneContext.GetNextWidget(widget));
                return widget;
            }

            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
            SceneContextBase& m_sceneContext;
            FEATSTD_MAKE_CLASS_UNCOPYABLE(SceneContextBaseWidgetCollectionBuilder);
        };

        bool AssetReaderBase<SceneContextBase>::ReadFirstPass(SceneContextBase& sceneContext, LoaderContext& context)
        {
            Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetSceneTheme(context.handle));
            if (!result.IsValid()) {
                FEATSTD_LOG_DEBUG("Scene theme AssetId not found!");
            }
            sceneContext.SetDefaultThemeId(AssetIdFunctions::GetLibraryId(result));
            sceneContext.SetLoadedThemeId(DefaultAssetProvider::GetInstance().GetCurrentThemeId());

            SceneContextBaseWidgetCollectionBuilder widgetCollectionBuilder(sceneContext);
            const AssetDataHandle& widgetCollectionHandle = CFFReader::GetSceneWidgets(context.handle);
            CffLoaderContext widgetCollectionContext = context.Clone(widgetCollectionHandle);
            if (!widgetCollectionBuilder.CreateAndBuildFirstPass(widgetCollectionContext)) {
                FEATSTD_LOG_ERROR("Failed to create widget collection of Scene" AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            return true;
        }

        bool AssetReaderBase<SceneContextBase>::ReadSecondPass(SceneContextBase& sceneContext, LoaderContext& context)
        {
            SceneContextBaseWidgetCollectionBuilder widgetCollectionBuilder(sceneContext);
            const AssetDataHandle& widgetCollectionHandle = CFFReader::GetSceneWidgets(context.handle);
            CffLoaderContext widgetCollectionContext = context.Clone(widgetCollectionHandle);
            if (!widgetCollectionBuilder.BuildSecondPass(widgetCollectionContext)) {
                FEATSTD_LOG_ERROR("Failed to load widget collection of Scene" AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }
            bool result = true;
            Int32 animationCount = CFFReader::GetSceneSceneAnimationInfoCount(context.handle);
            for (Int animationIndex = 0; animationIndex < animationCount; ++animationIndex) {
                const AssetDataHandle& aiHandle = CFFReader::GetSceneSceneAnimationInfoElementAt(context.handle, animationIndex);
                AssetId animationId = AssetIdFunctions::GetAssetId(CFFReader::GetSceneAnimationInfoAnimation(aiHandle));
                if (!animationId.IsValid()) {
                    FEATSTD_LOG_DEBUG("Animation AssetId not found!");
                }
                MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> resultAnimation = context.provider->GetAnimationByAssetId(animationId);
                if (resultAnimation.PointsToNull()){
                    FEATSTD_LOG_DEBUG("GetAnimationByAssetId returns a null pointer.");
                }
                Animation::AnimationPlayer::SharedPointer animation = Dynamic_Cast<Animation::AnimationPlayer::SharedPointer>(resultAnimation);
                LoaderContext aiContext = context.Clone(aiHandle);
                if (!SceneContextAnimationBlendedPropertyListAssetReader(sceneContext, animationId, animation).Read(aiContext)) {
                    FEATSTD_LOG_ERROR("Failed to create animation properties of Scene" AssetIdLogStr, AssetIdLogArgs(context.id));
                    result = false;
                }
            }

            return result;
        }

        void AssetReaderBase<SceneContextBase>::Dispose(SceneContextBase* sceneContext)
        {
            WidgetBase* widget = 0;
            for (bool isWidgetValid = sceneContext->GetFirstWidget(widget); isWidgetValid; isWidgetValid = sceneContext->GetNextWidget(widget)) {
                AssetBuilder<WidgetBase*>::Dispose(widget);
            }

            for (UInt32 animationIndex = 0; animationIndex < sceneContext->GetAttachedAnimationCount(); ++animationIndex) {
                Id animationId = sceneContext->GetAttachedAnimationId(animationIndex);
                Animation::AnimationPlayer::SharedPointer animation(Dynamic_Cast<Animation::AnimationPlayer*>(DefaultAssetProvider::GetInstance().GetAnimationById(animationId).GetPointerToSharedInstance()));
                if (animation != 0) {
                    AnimationController::SharedPointer animationController = animation->GetController();
                    if (animationController != 0) {
                        for (UInt32 propertyIndex = 0; propertyIndex < sceneContext->GetAnimatedPropertyAttachmentCount(animationId); ++propertyIndex) {
                            AnimationBlendedProperty::SharedPointer property = sceneContext->GetAnimatedPropertyAttachment(animationId, propertyIndex);
                            if (property != 0) {
                                for (SizeType index = 0; index < animationController->GetNumberOfProperties(); ++index) {
                                    if (property == animationController->GetProperty(index)) {
                                        static_cast<void>(animationController->RemoveProperty(index));
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        TemporaryTheme::TemporaryTheme(DefaultAssetProvider& provider, const AssetId& assetId):
            m_provider(provider),
            m_originalId(provider.GetCurrentThemeId())
        {
            if (assetId.IsValid()) {
                m_provider.SetCurrentThemeById(AssetIdFunctions::GetLibraryId(assetId));
            }
        }

        TemporaryTheme::~TemporaryTheme()
        {
            m_provider.SetCurrentThemeById(m_originalId);
        }

    }
}
