//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapAssetBuilder.h"
#include <Candera/EngineBase/Common/Bitmap.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BitmapCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDecompression.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultResourceProvider.h>
#include <Candera/System/Mathematics/Optional.h>

namespace Candera {
    namespace Internal {
        FEATSTD_LOG_SET_REALM(Candera::Diagnostics::LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<Bitmap>::ReadFirstPass(Bitmap& bitmap, LoaderContext& context)
        {
            if (bitmap.HasNext()) {
                const AssetDataHandle& nextBitmapHandle = CFFReader::GetBitmapNextBitmap(context.handle);
                LoaderContext nextContext = context.Clone(nextBitmapHandle);
                return ReadFirstPass(*(bitmap.GetNext().GetPointerToSharedInstance()), nextContext);
            }

            return true;
        }

        bool AssetReaderBase<Bitmap>::ReadSecondPass(Bitmap& bitmap, LoaderContext& context)
        {
            if (bitmap.HasNext()) {
                const AssetDataHandle& nextBitmapHandle = CFFReader::GetBitmapNextBitmap(context.handle);
                LoaderContext nextContext = context.Clone(nextBitmapHandle);
                return ReadSecondPass(*(bitmap.GetNext().GetPointerToSharedInstance()), nextContext);
            }

            return true;
        }

        Bitmap::SharedPointer AssetBuilderBase<Bitmap::SharedPointer>::Create(LoaderContext& context)
        {
            context.id = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!context.id.IsValid()) {
                FEATSTD_LOG_DEBUG("Bitmap AssetId is not valid");
            }
            return Create(context, AssetIdFunctions::GetLibraryId(context.id));
        }

        Bitmap::SharedPointer AssetBuilderBase<Bitmap::SharedPointer>::Create(LoaderContext& context, Id id)
        {
            const ResourceDataHandle& pixelsHandle =  DefaultResourceProvider::GetResourceProviderInstance().CreateRegionDataHandle(
                BitmapLib,
                id,
                static_cast<UInt16>(CFFReader::GetBitmapBitmapPixelsRegion(context.handle)),
                static_cast<UInt32>(CFFReader::GetBitmapBitmapPixelsLength(context.handle)));

            if (!pixelsHandle.m_accessor.IsValid()) {
                FEATSTD_LOG_ERROR("ResourceDataHandle for bitamp pixels cannot be initialized with current asset configuration. "
                    "Bitmap creation is skipped for object with AssetId: " AssetIdLogStr, AssetIdLogArgs(context.id));
                return Bitmap::SharedPointer(0);
            }

            Bitmap::SharedPointer result = Bitmap::Create(
                CFFReader::GetBitmapWidth(context.handle),
                CFFReader::GetBitmapHeight(context.handle),
                static_cast<Int>(CFFReader::GetBitmapPixelFormat(context.handle)),
                static_cast<Bitmap::PackAlignment>(CFFReader::GetBitmapPackAlignment(context.handle)),
                pixelsHandle,
                CFFReader::GetBitmapIsVerticallyFlipped(context.handle));

            if (result != 0)
            {
                if (CFFReader::GetBitmapBitmapLayout(context.handle) == 1)
                {
                    Int32 left = 0;
                    Int32 top = 0;
                    Int32 right = 0;
                    Int32 bottom = 0;
                    Bitmap::NinePatchProperties npp;
                    CFFReader::GetBitmapNinePatchStretchableArea(context.handle, left, top, right, bottom);
                    npp.left = static_cast<Int16>(left);
                    npp.top = static_cast<Int16>(top);
                    npp.right = static_cast<Int16>(right);
                    npp.bottom = static_cast<Int16>(bottom);
                    result->SetNinePatchproperties(Optional<Bitmap::NinePatchProperties>(npp));
                }
                if (CFFReader::GetBitmapIsPivotHintEnabled(context.handle))
                {
                    Float x = 0.0F;
                    Float y = 0.0F;
                    CFFReader::GetBitmapPivotHint(context.handle, x, y);
                    result->SetCenter(Optional<Vector2>(Vector2(x, y)));
                }

                const AssetDataHandle& nextBitmapHandle = CFFReader::GetBitmapNextBitmap(context.handle);
                if (nextBitmapHandle.IsValid()) {
                    LoaderContext nextContext = context.Clone(nextBitmapHandle);
                    result->SetNext(Create(nextContext, id));
                }
            }

            return result;
        }
    }
}
