//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationGroupAssetBuilder.h"
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnimationGroupCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/AnimationNodeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ControlNodeAnimationNodeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/CompositeGroupFinder.h>

#ifdef CANDERA_3D_ENABLED
#include <CanderaAssetLoader/AssetLoader3D/SceneContext.h>
#include <Candera/Engine3D/Core/TreeTraverser.h>
#include <Candera/Engine3D/Core/CompositeGroup.h>
#include <Candera/Engine3D/Core/Scene.h>
#endif
#ifdef CANDERA_2D_ENABLED
#include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
#include <Candera/Engine2D/Core/TreeTraverser2D.h>
#include <Candera/Engine2D/Core/CompositeGroup2D.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#endif

namespace Candera {
    using namespace Animation;
    using namespace Diagnostics;
    using namespace MemoryManagement;


    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);
        
        AnimationPlayerBase::SharedPointer AssetReaderBase<Animation::AnimationGroupPlayer>::GetAnimationNode(DefaultAssetProvider& assetProvider, const AssetDataHandle& handle)
        {
            const AssetId& playerId = AssetIdFunctions::GetAssetId(CFFReader::GetAnimationNodeAnimation(handle));
            if (playerId.IsValid()) {
                MemoryManagement::SharedPointer<Animation::AnimationPlayerBase> result = assetProvider.GetAnimationByAssetId(playerId);
                if (result.PointsToNull()){
                    FEATSTD_LOG_DEBUG("GetAnimationByAssetId returns a null pointer.");
                }
                return result;
            }
            else {
                FEATSTD_LOG_ERROR("Invalid AssetId during AnimationNode resolval");
            }

            return AnimationPlayerBase::SharedPointer(0);
        }

        template<typename TScene, typename TSceneContext, typename TNode, typename TCompositeGroup>
        AnimationPlayerBase::SharedPointer GetCompositeAnimationNodeInScene(TSceneContext* sceneContext, const AssetDataHandle& handle)
        {
            AnimationPlayerBase::SharedPointer result = AnimationPlayerBase::SharedPointer(0);

            Int32 pathLength = CFFReader::GetControlNodeAnimationNodeChainLength(handle);
            if (pathLength < 1) {
                FEATSTD_LOG_ERROR("Path length to short!");
                return result;
            }

            if (sceneContext != 0) {
                TScene* scene = sceneContext->GetScene();
                if (scene != 0) {
                    TCompositeGroup* compositeGroup = CompositeGroupFinder<TNode, TCompositeGroup>::GetCompositeGroup(scene, CFFReader::GetControlNodeAnimationNodeChainLength(handle), CFFReader::GetControlNodeAnimationNodeControlNodes(handle));
                    if (compositeGroup != 0) {
                        const AssetId& animationId = AssetIdFunctions::GetAssetId(CFFReader::GetControlNodeAnimationNodeCompositeAnimation(handle));
                        if (animationId.IsValid()) {
                            result = compositeGroup->GetAnimation(AssetIdFunctions::GetLibraryId(animationId));
                        }
                        else {
                            FEATSTD_LOG_DEBUG("GetAssetId is not valid");
                        }
                    }
                }
            }

            return result;
        }

        AnimationPlayerBase::SharedPointer AssetReaderBase<Animation::AnimationGroupPlayer>::GetCompositeAnimationNode(DefaultAssetProvider& assetProvider, const AssetDataHandle& handle)
        {
            AnimationPlayerBase::SharedPointer result = AnimationPlayerBase::SharedPointer(0);

            const AssetId& sceneId = AssetIdFunctions::GetAssetId(CFFReader::GetControlNodeAnimationNodeControlNodesElementAt(handle, 0));
            if (sceneId.IsValid()) {
                switch (sceneId.m_libraryType) {
                case SceneLib:
#ifdef CANDERA_3D_ENABLED
                {
                        SceneContext* resultSC=assetProvider.GetSceneByAssetId(sceneId);
                        if(resultSC == 0) {
                            FEATSTD_LOG_DEBUG("GetSceneByAssetId returns 0.");
                        }
                        result = GetCompositeAnimationNodeInScene<Scene, SceneContext, Node, CompositeGroup>(resultSC, handle);
                }
#endif
                        break;
                case Scene2DLib:
#ifdef CANDERA_2D_ENABLED
                {
                        Scene2DContext* resultGetScene = assetProvider.GetScene2DByAssetId(sceneId);
                        if (resultGetScene == 0){
                            FEATSTD_LOG_DEBUG("GetScene2DByAssetId returns 0.");
                        }
                        result = GetCompositeAnimationNodeInScene<Scene2D, Scene2DContext, Node2D, CompositeGroup2D>(resultGetScene, handle);
                }
#endif
                        break;
                    case CompositeLib:
                        FEATSTD_LOG_WARN("Animation Groups must not contain CompositeAnimations taken from Composite definition. Please use a CompositeGroup instance placed in a Scene.");
                        break;
                    case Composite2DLib:
                        FEATSTD_LOG_WARN("Animation Groups must not contain CompositeAnimations taken from Composite definition. Please use a CompositeGroup instance placed in a Scene.");
                        break;
                    default:
                        FEATSTD_DEBUG_FAIL();
                        break;
                }
            }
            else {
                FEATSTD_LOG_DEBUG("GetAssetId is not valid");
            }

            return result;
        }

        AnimationPlayerBase::SharedPointer AssetReaderBase<Animation::AnimationGroupPlayer>::GetAnimationNodeBase(DefaultAssetProvider& assetProvider, const AssetDataHandle& handle)
        {
            AnimationPlayerBase::SharedPointer result = AnimationPlayerBase::SharedPointer(0);
            switch (CFFReader::GetCanderaObjectItemType(handle)) {
                case AotAnimationNode:
                    result = GetAnimationNode(assetProvider, handle);
                    break;
                case AotCompositeNodeAnimationNode:
                    result = GetCompositeAnimationNode(assetProvider, handle);
                    break;
                default: 
                    FEATSTD_DEBUG_FAIL();
                    break;
            }

            return result;
        }

        bool AssetReaderBase<Animation::AnimationGroupPlayer>::ReadFirstPass(Animation::AnimationGroupPlayer& animationGroupPlayer, LoaderContext& context)
        {
            Int32 nodeCount = CFFReader::GetAnimationGroupPartialChildrenCount(context.handle);

            for (Int32 i = 0; i < nodeCount; i++) {
                const AssetDataHandle& animationPlayerHdl = CFFReader::GetAnimationGroupPartialChildrenElementAt(context.handle, i);
                if (!animationPlayerHdl.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read AnimationPlayerBase for AnimationGroupPlayer " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                const AnimationPlayerBase::SharedPointer& playerBase = GetAnimationNodeBase(*context.provider, animationPlayerHdl);
                if (playerBase != 0) {
                    AnimationPlayerBase::SharedPointer previousPlayerBase;

                    const AssetId& previousPlayerId = AssetIdFunctions::GetAssetId(CFFReader::GetAnimationNodeBasePrevious(animationPlayerHdl));
                    if (previousPlayerId.IsValid()) {
                        previousPlayerBase = context.provider->GetAnimationByAssetId(previousPlayerId);
                        if (previousPlayerBase == 0) {
                            FEATSTD_LOG_ERROR("Failed to retrieve previous AnimationPlayerBase AnimationGroupPlayer " AssetIdLogStr, AssetIdLogArgs(context.id));
                            return false;
                        }
                    }
                    else {
                        FEATSTD_LOG_DEBUG("GetAssetId is not valid");
                    }

                    animationGroupPlayer.AddPlayer(
                        playerBase,
                        previousPlayerBase,
                        static_cast<Animation::AnimationGroupPlayer::StartType>(CFFReader::GetAnimationNodeBaseStartType(animationPlayerHdl)),
                        CFFReader::GetAnimationNodeBaseDelay(animationPlayerHdl));
                }
                else {
                    FEATSTD_LOG_ERROR("Failed to retrieve AnimationPlayerBase in AnimationGroupPlayer " AssetIdLogStr, AssetIdLogArgs(context.id));
                }
            }

            return true;
        }

        MemoryManagement::SharedPointer<Animation::AnimationGroupPlayer> AssetBuilderBase<SharedPointer<Animation::AnimationGroupPlayer> >::Create(LoaderContext& /*context*/)
        {
            return Animation::AnimationGroupPlayer::Create();
        }
    }
}
