//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ARRAYDATATYPE_H)
#define CANDERA_ARRAYDATATYPE_H

#include <CanderaAssetLoader/AssetLoaderBase/ArrayDataTypeBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/ArrayProperty.h>

namespace Candera {
    namespace MetaInfo {
        namespace Internal {
            
            template<typename T>
            class TypedArrayDataTypeConverterToString: public ArrayDataTypeConverterToString {
                public:
                    TypedArrayDataTypeConverterToString(const ArrayProperty<T>& arrayProperty): m_arrayProperty(arrayProperty) {}

                    bool ConvertToString(Char *buf, UInt size) { return ArrayDataTypeConverterToString::ConvertToString(buf, size, static_cast<Int>(m_arrayProperty.GetCount())); }
                protected:
                    virtual bool ConvertItemToString(Char *buf, UInt size, Int index);

                private:
                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
                    const ArrayProperty<T>& m_arrayProperty;

                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                    FEATSTD_MAKE_CLASS_UNCOPYABLE(TypedArrayDataTypeConverterToString<T>);
            };

            template<typename T>
            bool TypedArrayDataTypeConverterToString<T>::ConvertItemToString(Char *buf, UInt size, Int index) 
            { 
                return Internal::DataType<T>::ConvertToString(m_arrayProperty.Get(index), buf, size); 
            }

            template<typename T>
            class TypedArrayDataTypeConverterFromString: public ArrayDataTypeConverterFromString {
                public:
                    TypedArrayDataTypeConverterFromString(ArrayProperty<T>& arrayProperty): m_arrayProperty(arrayProperty) {}

                protected:
                    virtual void Reserve(Int count);
                    virtual bool ConvertItemFromString(const Char* buf);

                private:
                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
                    ArrayProperty<T>& m_arrayProperty;

                    CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_INSTANCESOBTAINABLE)
                    FEATSTD_MAKE_CLASS_UNCOPYABLE(TypedArrayDataTypeConverterFromString<T>);
            };

            
            template<typename T>
            void TypedArrayDataTypeConverterFromString<T>::Reserve(Int count)
            {
                m_arrayProperty.Clear();
                m_arrayProperty.Reserve(count);
            }

            template<typename T>
            bool TypedArrayDataTypeConverterFromString<T>::ConvertItemFromString(const Char* buf)
            {
                T propVal;
                if (!Candera::MetaInfo::Internal::DataType<T>::ConvertFromString(propVal, buf)) {
                    return false;
                }
                m_arrayProperty.Add(propVal);
                return true;
            }

            /**
             * DataType specialization for array properties.
             */
            template<typename T> struct DataType<ArrayProperty<T> > {
                /**
                 * Convert an array property to string.
                 *
                 * The format is: [valueCount]:[valueCharCount];[value];[valueCharCount];[value]... where
                 *  valueCount and valueCharCount are integral numbers and value is the string representation
                 *  of the value, generated using DataType<T>::ConvertToString.
                 *
                 * @param val ArrayProperty to convert.
                 * @param buf Buffer to write the string representation.
                 * @param size Buffer length.
                 * @return true if conversion successful, false otherwise.
                 */
                template <typename P>
                static bool ConvertToString(const ArrayProperty<T>& val, Char *buf, UInt size) {
                    return Internal::TypedArrayDataTypeConverterToString<T>(val).ConvertToString(buf, size);
                }
                static bool ConvertToString(const ArrayProperty<T>& val, Char *buf, UInt size) {
                    return Internal::TypedArrayDataTypeConverterToString<T>(val).ConvertToString(buf, size);
                }

                /**
                 * Convert the string representation to an array property.
                 *
                 * @param val converted ArrayProperty.
                 * @param buf Buffer that contains the string representation.
                 * @return true if conversion successful, false otherwise.
                 */
                template <typename P>
                static bool ConvertFromString(ArrayProperty<T>& val, const Char *buf) {
                    return Internal::TypedArrayDataTypeConverterFromString<T>(val).ConvertFromString(buf);
                }
                static bool ConvertFromString(ArrayProperty<T>& val, const Char *buf) {
                    return Internal::TypedArrayDataTypeConverterFromString<T>(val).ConvertFromString(buf);
                }

                /**
                 * Get the PropertyEditor name as known by SceneComposer.
                 *
                 * @return string containing the SceneComposer property editor name.
                 */
                static const Char* GetEditor() {
                    return Internal::DataType<T>::GetEditor();
                }
            };
        }

        template<typename T> struct DataTypeDimensions<ArrayProperty<T> >{ enum { Count = DataTypeDimensions<T>::Count + 1 }; };

    }
}
#endif // CANDERA_ARRAYDATATYPE_H
