//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(CANDERA_ANIMATIONPROPERTYSETTERFACTORY_H)
#define CANDERA_ANIMATIONPROPERTYSETTERFACTORY_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/Animation/AnimationPropertySetter.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/MetaInfo/MetaInfo.h>
#include <Candera/System/Mathematics/Math.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetId.h>

namespace Candera {
        // Forward declarations
        namespace Animation {
            class AnimationPropertySetter;
        }
        class AssetProvider;
        class Node;
        class Node2D;

        namespace Internal {

        enum AnimatedPropertyChannels
        {
            apcInvalid = -1,    ///< Invalid.
            apcNone,            ///< None.
            apcDefault,         ///< Property is simple type (Float, Int32, bool, string)
            apcX,               ///< X
            apcY,               ///< Y
            apcZ,               ///< Z
            apcRed,             ///< Red
            apcGreen,           ///< Green
            apcBlue,            ///< Blue
            apcAlpha,           ///< Alpha
            apcLeft,            ///< Left
            apcTop,             ///< Top
            apcWidth,           ///< Width
            apcHeight,           ///< Height
            apcRight,            ///< Right
            apcBottom,            ///< Bottom
            apcU, // < not used yet in CreatePropertyMetaInfoSetter switch
            apcV, // < not used yet in CreatePropertyMetaInfoSetter switch
            apcW, ///
            // > not used yet in CreatePropertyMetaInfoSetter switch
            apcM_0_0,
            apcM_0_1,
            apcM_0_2,
            apcM_0_3,
            apcM_1_0,
            apcM_1_1,
            apcM_1_2,
            apcM_1_3,
            apcM_2_0,
            apcM_2_1,
            apcM_2_2,
            apcM_2_3,
            apcM_3_0,
            apcM_3_1,
            apcM_3_2,
            apcM_3_3
        };

        enum AnimationKeyframeType
        {
            aktUndef = 0,   ///< Keyframe type undefined.
            aktFloat,       ///< Float
            aktInt,         ///< Int
            aktBool,        ///< Bool
            aktString,      ///< String
            aktUInt,        ///< UInt
            aktInt16,       ///< Int16
            aktUInt16       ///< UInt16
        };

        enum PropertySetterType
        {
            pstAbsolute = 0,    ///< Absolute property setter.
            pstRelative         ///< Relative property setter.
        };

        /* Ids of the Animation Property Setters used in the asset.
           Please note that the value of the enums must NOT be changed
           without the agreement of the Scene Composer team!
           If you need to introduce a new enum value add it to the end of a section
           to assure backward compatibility.
         */
        enum AnimationPropertySetterId
        {
            // 3D property setters
            aps3dRotate             = 0,    ///< 3d Rotation
            aps3dScale              = 1,    ///< 3d Scale
            aps3dTranslate          = 2,    ///< 3d Translation

            aps3dAlpha              = 3,    ///< 3d Alpha
            aps3dRenderingEnabled   = 4,    ///< 3d Rendering enabled flag
            aps3dMatAmbient         = 5,    ///< 3d Ambient material
            aps3dMatDiffuse         = 6,    ///< 3d Diffuse material
            aps3dMatEmissive        = 7,    ///< 3d Emissive material
            aps3dMatSpecular        = 8,    ///< 3d Specular material
            aps3dMatSpecularPower   = 9,    ///< 3d Specular power

            aps3dLightAmbient       = 10,   ///< 3d Ambient light
            aps3dLightDiffuse       = 11,   ///< 3d Diffuse light
            aps3dLightSpecular      = 12,   ///< 3d Specular light

            aps3dFov                = 13,   ///< 3d Field of view
            aps3dLodCriterion       = 14,   ///< 3d Lod criterion

            aps3dUniform            = 15,   ///< 3d Uniform
            aps3dMorphWeight        = 16,   ///< 3d Morph weight

            aps3dCameraViewport = 17,
            aps3dCameraScissorRectangle = 18,
            aps3dCanvasRenderableVertexRectangle = 19,

            // 2D property setters
            aps2dRotate             = 50,   ///< 2d Rotation
            aps2dScale              = 51,   ///< 2d Scale
            aps2dTranslate          = 52,   ///< 2d Translation
            aps2dIsRenderingEnabled = 53,   ///< 2d Rendering enabled flag
            aps2dAlpha              = 54,   ///< 2d Alpha
            aps2dEffectProperty     = 55,   ///< 2d Effect property
            aps2dCompositeProperty  = 56,   ///< 2d Composite property
            aps2dIsSnapToDevicePixelEnabled = 57, ///< 2d SnapToDevicePixel property
            aps2dCamera2DViewport = 58,     ///< 2d Camera2D viewport property
            aps2dCameraScissorRectangle = 59, /// < 2d Camera2D ScissorRectangle  property
            aps2dNode2DBoundingRectangle = 60, /// < 2d Camera2D  BoundingRectangle  property
            // Refactored: Is now a common property
            apsCommonNodeLayoutMargin = 61, /// < Common Node + Node2D Margin  property for Layout
            // End refactoring

            // 2D property setters
            aps2dNode2DClippingRectangle = 62, /// < 2d Camera2D  BoundingRectangle  property

            // Common property setters
            apsWidgetProperty      = 100,   ///< Widget Property
            apsRenderTargetProperty = 101   ///< RenderTarget Property

            // New ids shall be added below:
            // Note: the new ids shall also be included in GetCategory() implementation
        };

        enum AnimationPropertySetterCategory {
            apsCategoryCommon,  ///< Common property setters
            apsCategory2d,      ///< 2D property setters
            apsCategory3d       ///< 3D property setters
        };

        struct PropertySetterAssociation {
            AnimationPropertySetterId   propertyId;
            Int32                       channelCount;
            AnimatedPropertyChannels    channel0Id;
            Int32                       setterClassId;
        };

        /**
        *  @brief AnimationPropertySetterFactory
        */

        class AnimationPropertySetterFactory
        {
        public:
            /**
             *  Sets the asset provider.
             *  @param assetProvider The asset provider that is set.
             */
            static void SetAssetProvider(Candera::AssetProvider* assetProvider);

            /**
             *  Create corresponding setter
             *  @param propertyId   The property Id
             *  @param propertyName The property Name
             *  @param apcList      The AnimatedPropertyChannels List
             *  @param setterType   The setter type default value pstAbsolute
             *  @return A Pointer to the created AnimationPropertySetter
             */
            static Candera::Animation::AnimationPropertySetter::SharedPointer CreateCorrespondingSetter(AnimationPropertySetterId propertyId, const Char* propertyName, const Internal::Vector<AnimatedPropertyChannels>& apcList, Candera::Internal::AssetId objectId, PropertySetterType setterType = pstAbsolute);

            template<typename T, typename P>
            static Candera::Animation::AnimationPropertySetter::SharedPointer CreatePropertyMetaInfoSetter(const MetaInfo::ObjectMetaInfo<T, P>* objectMetaInfo, T* object, const Char* propName, const Internal::Vector<AnimatedPropertyChannels>& apcList)
            {
                if ((objectMetaInfo == 0) || (object == 0) || (propName == 0)) {
                    return Animation::AnimationPropertySetter::SharedPointer(0);
                }

                P* propMetaInfo = objectMetaInfo->LookupItem(propName);
                if (propMetaInfo == 0) {
                    return Animation::AnimationPropertySetter::SharedPointer(0);
                }

                Int channels[4];
                SizeType channelCount = Math::Minimum(apcList.Size(), static_cast<SizeType>(4));

                for (SizeType i = 0; i < channelCount; ++i) {
                    switch (apcList[i]) {
                        case apcInvalid:
                        case apcNone:
                        case apcDefault:
                        case apcX:
                        case apcRed:
                        case apcLeft:
                            channels[i] = 0;
                            break;
                        case apcY:
                        case apcGreen:
                        case apcTop:
                            channels[i] = 1;
                            break;
                        case apcZ:
                        case apcBlue:
                        case apcWidth:
                        case apcRight:
                            channels[i] = 2;
                            break;
                        case apcAlpha:
                        case apcHeight:
                        case apcBottom:
                        case apcW:
                            channels[i] = 3;
                            break;
                        default:
                            FEATSTD_DEBUG_FAIL();
                            break;
                    }
                }

                return propMetaInfo->CreateAnimationPropertySetter(object, static_cast<Int>(apcList.Size()), channels);
            }

            /**
             * @brief Returns the corresponding category of the specified animation property setter id
             */
            static AnimationPropertySetterCategory GetCategory(AnimationPropertySetterId apsId);
        private:

            static Candera::AssetProvider* m_assetProvider;

        };
    }
}
#endif // CANDERA_ANIMATIONPROPERTYSETTERFACTORY_H

