//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "AnimationPropertySetterFactory.h"
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>

#ifdef CANDERA_2D_ENABLED
    #include <CanderaAssetLoader/AssetLoader2D/AnimationPropertySetterFactory2D.h>
#endif //CANDERA_2D_ENABLED
#ifdef CANDERA_3D_ENABLED
    #include <CanderaAssetLoader/AssetLoader3D/AnimationPropertySetterFactory3D.h>
#endif //CANDERA_3D_ENABLED
#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Animation/LayoutMarginsPropertySetter.h>
#endif // defined(CANDERA_LAYOUT_ENABLED)

#include <Candera/System/MetaInfo/PropertyMetaInfoAnimationPropertySetters.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageDescriptor.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitMetaInfo.h>
#include <CanderaWidget/WidgetBase/WidgetBase.h>
#include <CanderaAssetLoader/AssetLoaderBase/AnimationPropertySetterFactoryCommon.h>

namespace Candera {
    namespace Internal {

        FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

        AssetProvider* AnimationPropertySetterFactory::m_assetProvider = 0;

        Animation::AnimationPropertySetter::SharedPointer AnimationPropertySetterFactory::CreateCorrespondingSetter(AnimationPropertySetterId propertyId, const Char* propertyName, const Vector<AnimatedPropertyChannels>& apcList, AssetId objectId, PropertySetterType setterType)
        {
            AnimationPropertySetterCategory apsCategory = GetCategory(propertyId);

            if (apsCategory == apsCategoryCommon) {
                if (propertyId == apsWidgetProperty) {
                    WidgetBase* widget = m_assetProvider->GetWidgetByAssetId(objectId);
                    if (widget != 0) {
                        return CreatePropertyMetaInfoSetter(widget->GetMetaInfo(), widget, propertyName, apcList);
                    }
                    else {
                        FEATSTD_LOG_DEBUG("Widget not found.");
                    }
                }
                else if (propertyId == apsRenderTargetProperty) {
                    GraphicDeviceUnit* gdu = m_assetProvider->GetGraphicDeviceUnitByAssetId(objectId);
                    if (gdu != 0) {
                        return CreatePropertyMetaInfoSetter(DevicePackageDescriptor::GetMetaInformation(gdu->GetUnitType()), static_cast<DeviceMetaInfoHost*>(gdu), propertyName, apcList);
                    }
                    else {
                        FEATSTD_LOG_DEBUG("GetGraphicDeviceUnit not found.");
                    }
                }
                else if (propertyId == apsCommonNodeLayoutMargin) {
                    return AnimationPropertySetterFactoryCommon::CreateCorrespondingSetter(propertyId, propertyName, apcList, objectId, setterType);
                }
                else {
                    //do nothing
                }
            }

    #ifdef CANDERA_3D_ENABLED
            if (apsCategory == apsCategory3d) {
                return AnimationPropertySetterFactory3D::CreateCorrespondingSetter(propertyId, propertyName, apcList, objectId, setterType);
            }
    #endif //CANDERA_3D_ENABLED

    #ifdef CANDERA_2D_ENABLED
            if (apsCategory == apsCategory2d) {
                return AnimationPropertySetterFactory2D::CreateCorrespondingSetter(propertyId, propertyName, apcList, objectId, setterType);
            }
    #endif //CANDERA_2D_ENABLED

            return Animation::AnimationPropertySetter::SharedPointer(0);
        }

        void AnimationPropertySetterFactory::SetAssetProvider(AssetProvider* assetProvider)
        {
    #ifdef CANDERA_2D_ENABLED
            AnimationPropertySetterFactory2D::SetAssetProvider(assetProvider);
    #endif //CANDERA_2D_ENABLED
    #ifdef CANDERA_3D_ENABLED
            AnimationPropertySetterFactory3D::SetAssetProvider(assetProvider);
    #endif //CANDERA_3D_ENABLED
            AnimationPropertySetterFactoryCommon::SetAssetProvider(assetProvider);
            m_assetProvider = assetProvider;
        }

        Candera::Internal::AnimationPropertySetterCategory AnimationPropertySetterFactory::GetCategory(AnimationPropertySetterId apsId)
        {
            AnimationPropertySetterCategory result = apsCategoryCommon;
            switch (apsId) {
                case aps3dRotate:
                case aps3dScale:
                case aps3dTranslate:
                case aps3dAlpha:
                case aps3dRenderingEnabled:
                case aps3dMatAmbient:
                case aps3dMatDiffuse:
                case aps3dMatEmissive:
                case aps3dMatSpecular:
                case aps3dMatSpecularPower:
                case aps3dLightAmbient:
                case aps3dLightDiffuse:
                case aps3dLightSpecular:
                case aps3dFov:
                case aps3dLodCriterion:
                case aps3dUniform:
                case aps3dMorphWeight:
                case aps3dCameraViewport:
                case aps3dCameraScissorRectangle:
                case aps3dCanvasRenderableVertexRectangle:
                    result = apsCategory3d;
                    break;
                case aps2dRotate:
                case aps2dScale:
                case aps2dTranslate:
                case aps2dIsRenderingEnabled:
                case aps2dIsSnapToDevicePixelEnabled:
                case aps2dAlpha:
                case aps2dEffectProperty:
                case aps2dCamera2DViewport:
                case aps2dCameraScissorRectangle:
                case aps2dNode2DBoundingRectangle:
                case aps2dNode2DClippingRectangle:
                    result = apsCategory2d;
                    break;
                case apsCommonNodeLayoutMargin:
                case apsWidgetProperty:
                case apsRenderTargetProperty:
                    result = apsCategoryCommon;
                    break;
                default:
                    FEATSTD_LOG_ERROR("Could not determine the category of the animation property setter. Unknown aps id");
                    break;
            }
            return result;
        }
    }
}

