//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "StereoCameraAssetBuilder.h"
#include <Candera/Engine3D/Core/StereoCamera.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Rtti/Rtti.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/StereoCameraCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

    namespace Internal {

        StereoCamera* AssetBuilderBase<StereoCamera*>::Create(LoaderContext& /*context*/)
        {
            return StereoCamera::Create();
        }

        bool AssetReaderBase<StereoCamera>::ReadFirstPass(StereoCamera& stereoscopicCamera, LoaderContext& context)
        {
            stereoscopicCamera.SetConvergenceDistance(CFFReader::GetStereoCameraConvergenceDistance(context.handle));
            stereoscopicCamera.SetEyeSeparation(CFFReader::GetStereoCameraEyeSeparation(context.handle));

            return true;
        }

        bool AssetReaderBase<StereoCamera>::ReadSecondPass(StereoCamera& stereoscopicCamera, LoaderContext& context)
        {
            const AssetId& leftEyeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetStereoCameraLeftEye(context.handle));
            if (!leftEyeAssetId.IsValid()) {
                FEATSTD_LOG_WARN("StereoCamera " AssetIdLogStr " has no left eye Camera attached", AssetIdLogArgs(context.id));
                return false;
            }
            Node* result = context.provider->GetNodeByAssetId(leftEyeAssetId);
            Camera* leftEyeCamera = Dynamic_Cast<Camera*>(result);
            if (leftEyeCamera != 0) {
                if (!stereoscopicCamera.SetLeftEye(leftEyeCamera)) {
                    FEATSTD_LOG_WARN("Could not set left eye Camera " AssetIdLogStr " for StereoCamera " AssetIdLogStr, AssetIdLogArgs(leftEyeAssetId), AssetIdLogArgs(context.id));
                    return false;
                }
            }
            else {
                FEATSTD_LOG_ERROR("Failed to retrieve left eye Camera for StereoCamera " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            const AssetId& rightEyeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetStereoCameraRightEye(context.handle));
            if (!rightEyeAssetId.IsValid()) {
                FEATSTD_LOG_WARN("StereoCamera " AssetIdLogStr " has no right eye Camera attached", AssetIdLogArgs(context.id));
                return false;
            }
            Node* node = context.provider->GetNodeByAssetId(rightEyeAssetId);
            Camera* rightEyeCamera = Dynamic_Cast<Camera*>(node);
            if (rightEyeCamera != 0) {
                if (!stereoscopicCamera.SetRightEye(rightEyeCamera)) {
                    FEATSTD_LOG_WARN("Could not set right eye Camera " AssetIdLogStr " for StereoCamera " AssetIdLogStr, AssetIdLogArgs(rightEyeAssetId), AssetIdLogArgs(context.id));
                    return false;
                }
            }
            else {
                FEATSTD_LOG_ERROR("Failed to retrieve right eye Camera for StereoCamera " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            return true;
        }
    }
}
