//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ShaderAssetBuilder.h"

#include <FeatStd/Util/StaticObject.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/ShaderProgramCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDecompression.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultResourceProvider.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>

namespace Candera { namespace Internal {

    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaAssetLoader);

    using namespace Candera::MemoryManagement;

    ShaderPtr ShaderBuilder::Create(LoaderContext& /*context*/)
    {
        return Shader::Create();
    }
    
    bool ShaderReader::ReadFirstPass(Shader& shader, LoaderContext& context)
    {
        const ResourceDataHandle& vertexShaderResourceHandle = DefaultResourceProvider::GetResourceProviderInstance().CreateRegionDataHandle(
            ShaderLib,
            shader.GetId(),
            static_cast<UInt16>(CFFReader::GetShaderProgramVertexShaderBytesRegion(context.handle)),
            static_cast<UInt32>(CFFReader::GetShaderProgramVertexShaderBytesLength(context.handle)));

        if (!vertexShaderResourceHandle.m_accessor.IsValid()) {
            FEATSTD_LOG_ERROR("Shader " AssetIdLogStr " has no vertex shader attached", AssetIdLogArgs(context.id));
            return false;
        }

        if (!shader.SetVertexShaderResourceHandle(vertexShaderResourceHandle)) {
            FEATSTD_LOG_ERROR("Vertex shader failed to be set to Shader" AssetIdLogStr, AssetIdLogArgs(context.id));
            return false;
        }

        Shader::BuildType shaderBuildType = Shader::ShaderSource;
        bool isShaderBuildTypeKnown = RenderDevice::GetShaderBuildType(vertexShaderResourceHandle, shaderBuildType);
        
        const ResourceDataHandle& fragmentShaderResourceHandle = DefaultResourceProvider::GetResourceProviderInstance().CreateRegionDataHandle(
            ShaderLib,
            shader.GetId(),
            static_cast<UInt16>(CFFReader::GetShaderProgramFragmentShaderBytesRegion(context.handle)),
            static_cast<UInt32>(CFFReader::GetShaderProgramFragmentShaderBytesLength(context.handle)));

        if (!fragmentShaderResourceHandle.m_accessor.IsValid()) {
            if (isShaderBuildTypeKnown) {
                if (shaderBuildType == Shader::ShaderProgramBinary) {
                    // Compiled shaders have the binary in the vertex shader region
                }
                else {
                    FEATSTD_LOG_ERROR("Shader " AssetIdLogStr " has no fragment shader attached", AssetIdLogArgs(context.id));
                    return false;
                }
            }
            else {
                FEATSTD_LOG_WARN("Shader build type of Shader " AssetIdLogStr " could not be checked. Invalid shader might be loaded.", AssetIdLogArgs(context.id));
            }
        }

        if (!shader.SetFragmentShaderResourceHandle(fragmentShaderResourceHandle)) {
            FEATSTD_LOG_ERROR("Fragment shader failed to be set to Shader" AssetIdLogStr, AssetIdLogArgs(context.id));
            return false;
        }

        shader.SetMaximumInstanceCount(CFFReader::GetShaderProgramMaxInstanceCount(context.handle));
        return true;
    }
}}
