//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SceneContextAssetBuilder.h"

#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoader3D/SceneContext.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/SceneCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetBuilders/WidgetBaseAssetBuilder.h>
#include <CanderaAssetLoader/AssetLoader3D/AssetBuilders/SceneAssetBuilder.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;
    using namespace MetaInfo;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<SceneContext>::ReadSecondPass(SceneContext& sceneContext, LoaderContext& context)
        {
            Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetSceneTheme(context.handle));
            if (!result.IsValid()) {
                FEATSTD_LOG_DEBUG("Scene Theme AssetId is not valid");
            }
            TemporaryTheme temporaryTheme(*(context.provider), (result));
            
            Scene* scene = sceneContext.GetScene();
            if ((scene == 0) || (!AssetReader<Scene>::ReadSecondPass(*scene, context))) {
                FEATSTD_LOG_WARN("Failed to load scene tree for Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            return true;
        }

        SceneContext* AssetBuilderBase<SceneContext*>::Create(LoaderContext& context)
        {
            context.id = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(context.handle));
            if (!context.id.IsValid()) {
                FEATSTD_LOG_DEBUG("AssetId is not valid");
            }
            SceneContext* sceneContext = ASSETLOADER_TRANSIENT_NEW(SceneContext);
            if (sceneContext == 0) {
                FEATSTD_LOG_WARN("Failed to allocate memory for SceneContext.");
                return 0;
            }

            Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetSceneTheme(context.handle));
            if (!result.IsValid()) {
                FEATSTD_LOG_DEBUG("Scene Theme AssetId is not valid");
            }
            TemporaryTheme temporaryTheme(*(context.provider), (result));
            
            if (context.reloadList == 0) {
                context.reloadList = &sceneContext->m_reloadList;
            }
            context.iterativeLoad = DefaultAssetProvider::GetInstance().IsIterativeLoadingEnabled();

            Node* node = AssetBuilder<Node*>::CreateAndBuildFirstPass(context);
            if (node == 0) {
                FEATSTD_LOG_WARN("Failed to create scene tree for Scene " AssetIdLogStr, AssetIdLogArgs(context.id));
                ASSETLOADER_DELETE(sceneContext);
                return 0;
            }

            Scene* scene = Dynamic_Cast<Scene*>(node);
            if (scene == 0) {
                FEATSTD_LOG_ERROR("Type described in asset differs from actual type for Node " AssetIdLogStr, AssetIdLogArgs(context.id));
                AssetBuilder<Node*>::Dispose(node);
                ASSETLOADER_DELETE(sceneContext);
                return 0;
            }

            sceneContext->SetScene(scene);

            return sceneContext;
        }

        void AssetBuilderBase<SceneContext*>::Dispose(SceneContext* sceneContext)
        {
            AssetReaderBase<SceneContextBase>::Dispose(sceneContext);
            AssetBuilder<Node*>::Dispose(sceneContext->GetScene());
            ASSETLOADER_DELETE(sceneContext);
        }

        bool AssetBuilderBase<SceneContext*>::IsValid(const SceneContext* sceneContext, LoaderContext& context)
        {
            Id currentThemeId = context.provider->GetCurrentThemeId();
            if (currentThemeId != 0) {
                return (sceneContext->GetLoadedThemeId() == currentThemeId);
            }

            return (sceneContext->GetLoadedThemeId() == sceneContext->GetDefaultThemeId());
        }
    }
}
