//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "MorphingMeshAssetBuilder.h"
#include <Candera/Engine3D/Core/MorphingMesh.h>
#include <Candera/Engine3D/Core/VertexBuffer.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/MorphingMeshCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/MorphItemCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        MorphingMesh* AssetBuilderBase<MorphingMesh*>::Create(LoaderContext& /*context*/)
        {
            return MorphingMesh::Create();
        }

        bool AssetReaderBase<MorphingMesh>::ReadFirstPass(MorphingMesh& morphingMesh, LoaderContext& context)
        {
            Int32 morphCount = Math::Minimum<Int32>(8, CFFReader::GetMorphingMeshMorphItemsCount(context.handle));

            SharedPointer<VertexBuffer> vertexBuffers[8];
            for (Int32 morphIndex = 0; morphIndex < morphCount; ++morphIndex) {
                const AssetDataHandle& morphHandle = CFFReader::GetMorphingMeshMorphItemsElementAt(context.handle, CFFReader::CFF_MORPH_ITEM_SIZE, morphIndex);
                if (!morphHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read VertexBuffer at index %d of MorphingMesh " AssetIdLogStr, morphIndex, AssetIdLogArgs(context.id));
                    return false;
                }
                const AssetId& vertexBufferAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetMorphItemVertexBuffer(morphHandle));
                if (!vertexBufferAssetId.IsValid()) {
                    FEATSTD_LOG_INFO("MorphingMesh " AssetIdLogStr " has no VertexBuffer attached at index %d", morphIndex, AssetIdLogArgs(context.id));

                }
                const VertexBuffer::SharedPointer& vertexBuffer = context.provider->GetVertexBufferByAssetId(vertexBufferAssetId);
                if (vertexBuffer == 0) {
                    FEATSTD_LOG_WARN("Failed to associate VertexBuffer " AssetIdLogStr " at index to MorphingMesh "
                        AssetIdLogStr, AssetIdLogArgs(vertexBufferAssetId), morphIndex, AssetIdLogArgs(context.id));
                }
                vertexBuffers[morphIndex] = vertexBuffer;
                morphingMesh.SetMorphWeight(morphIndex, CFFReader::GetMorphItemWeight(morphHandle));
            }

            const VertexBuffer::SharedPointer& vertexBuffer = MorphingMesh::CreateSplicedVertexBuffer(vertexBuffers, morphCount);
            if (vertexBuffer == 0) {
                FEATSTD_LOG_WARN("Failed to splice VertexBuffer MorphingMesh " AssetIdLogStr, AssetIdLogArgs(context.id));
            }
            morphingMesh.SetVertexBuffer(vertexBuffer);

            return true;
        }
    }
}
