//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "LodNodeAssetBuilder.h"
#include <Candera/Engine3D/Core/LodNode.h>
#include <Candera/Engine3D/Core/BlendLodRenderStrategy.h>
#include <Candera/Engine3D/Core/DiscreteLodRenderStrategy.h>
#include <Candera/Engine3D/Core/GenericValueLodCriterion.h>
#include <Candera/Engine3D/Core/DistanceToCameraLodCriterion.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LodNodeCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LodStrategyCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/BlendLodStrategyCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/GenericLodCriterionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/LodNodeItemInfoCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    using namespace Diagnostics;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        LodNode* AssetBuilderBase<LodNode*>::Create(LoaderContext& /*context*/)
        {
            return LodNode::Create();
        }

        void AssetBuilderBase<LodNode*>::Dispose(const LodNode* lodNode)
        {
            const LodRenderStrategy* strategy = lodNode->GetLodRenderStrategy();
            if (strategy != 0) {
                const LodCriterion* criterion = strategy->GetLodCriterion();
                if (criterion != 0) {
                    FEATSTD_DELETE(criterion);
                    //criterion = 0;    //Removed because of MISRA Rule 0-1-6
                }
                FEATSTD_DELETE(strategy);
                //strategy = 0;         //Removed because of MISRA Rule 0-1-6
            }
        }

        bool AssetReaderBase<LodNode>::ReadFirstPass(LodNode& lodNode, LoaderContext& context)
        {
            LodRenderStrategy* lodRenderStrategy = 0;
            const AssetDataHandle& lodNodeStrategyHandle = CFFReader::GetLodNodeStrategy(context.handle);
            if (!lodNodeStrategyHandle.IsValid()) {
                FEATSTD_LOG_ERROR("There is no LodRenderStrategy associated to LodNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            switch (CFFReader::GetLodStrategyStrategyType(lodNodeStrategyHandle)) {
                case LodStrategyTypeDiscrete:
                    lodRenderStrategy = FEATSTD_NEW(DiscreteLodRenderStrategy);
                    break;
                case LodStrategyTypeBlend: {
                    BlendLodRenderStrategy* blendLodRenderStrategy = FEATSTD_NEW(BlendLodRenderStrategy);
                    if (blendLodRenderStrategy != 0) {
                        blendLodRenderStrategy->SetDepthBiasEnabled(CFFReader::GetBlendLodStrategyIsDepthBiasEnabled(lodNodeStrategyHandle));
                        Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetBlendLodStrategyOpaqueROB(lodNodeStrategyHandle));
                        if (!result.IsValid()) {
                            FEATSTD_LOG_DEBUG("AssetId is not valid");
                        }
                        const Char* ROBName = context.provider->GetNameByAssetId(result);
                        if ((ROBName != 0) && (StringPlatform::CompareStrings(ROBName, "") == 0)) {
                            ROBName = 0;
                        }
                        blendLodRenderStrategy->SetOpaqueRenderOrderBinAssignment(ROBName);
                        result = AssetIdFunctions::GetAssetId(CFFReader::GetBlendLodStrategyTransparentROB(lodNodeStrategyHandle));
                        if (!result.IsValid()) {
                            FEATSTD_LOG_DEBUG("GetAssetId is not valid");
                        }
                        ROBName = context.provider->GetNameByAssetId(result);
                        if ((ROBName != 0) && (StringPlatform::CompareStrings(ROBName, "") == 0)) {
                            ROBName = 0;
                        }
                        blendLodRenderStrategy->SetTransparentRenderOrderBinAssignment(ROBName);
                    }
                    lodRenderStrategy = blendLodRenderStrategy;
                    break;
                }
                default:
                    FEATSTD_LOG_ERROR("Could not determine the strategy type associated to LodNode " AssetIdLogStr, AssetIdLogArgs(context.id));
            }

            if (lodRenderStrategy != 0) {
                const AssetDataHandle& lodNodeCriterionHandle = CFFReader::GetLodNodeCriterion(context.handle);
                if (!lodNodeCriterionHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("There is no LodCriterion associated to LodNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }

                switch (CFFReader::GetLodCriterionCriterionType(lodNodeCriterionHandle)) {
                    case LodCriterionTypeGeneric: {
                        GenericValueLodCriterion* genericValueLodCriterion = FEATSTD_NEW(GenericValueLodCriterion);
                        if (genericValueLodCriterion != 0) {
                            genericValueLodCriterion->SetCriterionValue(CFFReader::GetGenericLodCriterionCriterionValue(lodNodeCriterionHandle));
                        }
                        else {
                            FEATSTD_LOG_WARN("Could not create criterion for LodNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                            return false;
                        }
                        lodRenderStrategy->SetLodCriterion(genericValueLodCriterion);
                        break;
                    }
                    case LodCriterionTypeDistanceToCamera: {
                        DistanceToCameraLodCriterion* distanceToCameraLodCriterion = FEATSTD_NEW(DistanceToCameraLodCriterion);
                        if (distanceToCameraLodCriterion != 0) {
                            lodRenderStrategy->SetLodCriterion(distanceToCameraLodCriterion);
                        }
                        else {
                            FEATSTD_LOG_WARN("Could not create criterion for LodNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                            return false;
                        }
                        break;
                    }
                    default:
                        FEATSTD_LOG_ERROR("Could not determine the criterion type associated to LodNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                }
            }
            else {
                FEATSTD_LOG_ERROR("Could not create strategy for LodNode " AssetIdLogStr, AssetIdLogArgs(context.id));
            }

            lodNode.SetLodRenderStrategy(lodRenderStrategy);

            Node* lodLevelNode = lodNode.GetFirstChild();
            for (Int32 lodLevelIndex = 0; (lodLevelIndex < CFFReader::GetLodNodeLodNodeItemInfoListCount(context.handle)) && (lodLevelNode != 0); ++lodLevelIndex) {
                const AssetDataHandle& lodLevelHandle = CFFReader::GetLodNodeLodNodeItemInfoListElementAt(context.handle, CFFReader::CFF_LOD_NODE_ITEM_INFO_SIZE, lodLevelIndex);
                if (!lodLevelHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("There is no LodLevel no.%d associated to LodNode " AssetIdLogStr, lodLevelIndex, AssetIdLogArgs(context.id));
                    return false;
                }
                const Float lowerBound = CFFReader::GetLodNodeItemInfoLowerBound(lodLevelHandle);
                const Float upperBound = CFFReader::GetLodNodeItemInfoUpperBound(lodLevelHandle);
                if (!lodNode.SetLodLevel(static_cast<UInt>(lodLevelIndex), lodLevelNode, lowerBound, upperBound)) {
                    FEATSTD_LOG_ERROR("Setting LOD level no.%d failed to LodNode " AssetIdLogStr, lodLevelIndex, AssetIdLogArgs(context.id));
                }

                lodLevelNode = lodLevelNode->GetNextSibling();
            }

            return true;
        }
    }
}
