//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CanvasTextAssetBuilder.h"

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CanvasTextCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TextNode2DCffReader.h>
#include <Candera/Engine3D/Canvas/CanvasText.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

#include <CanderaWidget/WidgetBase/StringDataTypes.h>
namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;
    using namespace TextRendering;
    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        CanvasText* AssetBuilderBase<CanvasText*>::Create(LoaderContext& context)
        {
            FEATSTD_UNUSED(context);
            return CanvasText::Create();
        }

        bool AssetReaderBase<CanvasText>::ReadFirstPass(CanvasText& canvasText, const LoaderContext& context)
        {
            Int32 width;
            Int32 height;

            Float red = 0.0F;
            Float green = 0.0F;
            Float blue = 0.0F;
            Float alpha = 0.0F;

            const AssetId& styleAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCanvasTextStyle(context.handle));
            if (styleAssetId.IsValid()) {
                const SharedStyle::SharedPointer& style = DefaultAssetProvider::GetInstance().GetTextStyleByAssetId(styleAssetId);
                if (style == 0) {
                    FEATSTD_LOG_WARN("Style " AssetIdLogStr " not found for CanvasText " AssetIdLogStr,
                                     AssetIdLogArgs(styleAssetId), AssetIdLogArgs(context.id));
                }
                else {
                    canvasText.SetStyle(style);
                }
            }
            else {
                FEATSTD_LOG_WARN("CanvasText " AssetIdLogStr " has no style attached!", AssetIdLogArgs(context.id));
            }

            FeatStd::String convertedText;
            const Char* textInAsset = CFFReader::GetCanvasTextTranslatableText(context.handle);
            if (MetaInfo::DataType<FeatStd::String>::ConvertFromString(convertedText, textInAsset)) {
                canvasText.SetText(convertedText);
            }
            else {
                canvasText.SetText(FeatStd::String(textInAsset));
            }

            FeatStd::String convertedTruncationText;
            textInAsset = CFFReader::GetCanvasTextTranslatableTruncationText(context.handle);
            if (MetaInfo::DataType<FeatStd::String>::ConvertFromString(convertedTruncationText, textInAsset)) {
                canvasText.SetTruncationText(convertedTruncationText);
            }
            else {
                canvasText.SetTruncationText(FeatStd::String(textInAsset));
            }
            canvasText.SetMultiLineEnabled(CFFReader::GetCanvasTextIsMultiLineEnabled(context.handle));
            canvasText.SetWordWrapEnabled(CFFReader::GetCanvasTextIsWordWrapEnabled(context.handle));
            canvasText.SetLineSpacing(CFFReader::GetTextNode2DLineSpacing(context.handle));

            CFFReader::GetCanvasTextSize(context.handle, width, height);
            canvasText.SetSize(TextSize(static_cast<Int16>(width), static_cast<Int16>(height)));

            CFFReader::GetCanvasTextColor(context.handle, red, green, blue, alpha);
            canvasText.SetTextColor(Color(red, green, blue, alpha));

            Int32 enumValue = CFFReader::GetCanvasTextHorizontalTextAlignment(context.handle);
            if ((enumValue >= 0) && (enumValue <= CanvasText::HorizontalTextAlignment::Justified)) {
                canvasText.SetHorizontalTextAlignment(static_cast<CanvasText::HorizontalTextAlignment::Enum>(enumValue));
            }
            else {
                FEATSTD_LOG_WARN("CanvasText " AssetIdLogStr " has invalid text alignment attached!", AssetIdLogArgs(context.id));
            }

            enumValue = CFFReader::GetCanvasTextTruncationMode(context.handle);
            if ((enumValue >= 0) && (enumValue <= CanvasText::TruncationMode::CustomText)) {
                canvasText.SetTruncationMode(static_cast<CanvasText::TruncationMode::Enum>(enumValue));
            }
            else {
                FEATSTD_LOG_WARN("CanvasText " AssetIdLogStr " has invalid truncation mode attached!", AssetIdLogArgs(context.id));
            }

            return true;
        }
    }
}

