//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CameraAssetBuilder.h"
#include <Candera/Engine3D/Core/BenchmarkCameraRenderStrategy.h>
#include <Candera/Engine3D/Core/OcclusionCullingCameraRenderStrategy.h>
#include <Candera/Engine3D/Core/Camera.h>
#include <Candera/Engine3D/Core/Scene.h>
#include <Candera/Engine3D/Core/GenericProjection.h>
#include <Candera/Engine3D/Core/OrthographicProjection.h>
#include <Candera/Engine3D/Core/PerspectiveProjection.h>
#include <Candera/Engine3D/Core/SkyBox.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CameraCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<Camera>::ReadFirstPass(Camera& camera, LoaderContext& context)
        {
            const AssetId& grpahicDeviceUnitAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCameraRenderTarget(context.handle));
            if (!grpahicDeviceUnitAssetId.IsValid()) {
                FEATSTD_LOG_INFO("There is no GraphicDeviceUnit associated to Camera " AssetIdLogStr, AssetIdLogArgs(context.id));
            }
            else {
                GraphicDeviceUnit* grapahicDeviceUnit = context.provider->GetGraphicDeviceUnitByAssetId(grpahicDeviceUnitAssetId);

                if (grapahicDeviceUnit != 0) {
                    RenderTarget3D* renderTarget = grapahicDeviceUnit->ToRenderTarget3D(0);
                    if (renderTarget != 0) {
                        camera.SetRenderTarget(renderTarget);
                    }
                    else {
                        FEATSTD_LOG_WARN("GraphicDeviceUnit " AssetIdLogStr " cannot be used as a RenderTarget3D for Camera "
                            AssetIdLogStr, AssetIdLogArgs(grpahicDeviceUnitAssetId), AssetIdLogArgs(context.id));
                    }
                }
                else {
                    FEATSTD_LOG_WARN("GraphicDeviceUnit " AssetIdLogStr " could not be created during building Camera"
                        AssetIdLogStr, AssetIdLogArgs(grpahicDeviceUnitAssetId), AssetIdLogArgs(context.id));
                }
            }

            Float f1 = 0.0F;
            Float f2 = 0.0F;
            Float f3 = 0.0F;
            Float f4 = 0.0F;

            CFFReader::GetCameraViewport(context.handle, f1, f2, f3, f4);
            camera.SetViewport(Rectangle(f1, f2, f3, f4));

            CFFReader::GetCameraScissorRectangle(context.handle, f1, f2, f3, f4);
            camera.SetScissorRectangle(Rectangle(f1, f2, f3, f4));
            camera.SetScissoringEnabled(CFFReader::GetCameraIsScissoringEnabled(context.handle));
            camera.SetSequenceNumber(CFFReader::GetCameraSequenceNumber(context.handle));
            camera.SetViewingFrustumCullingEnabled(CFFReader::GetCameraIsViewingFrustumCullingEnabled(context.handle));
            camera.SetSwapEnabled(CFFReader::GetCameraIsSwapEnabled(context.handle));

            switch (static_cast<CameraProjectionType>(CFFReader::GetCameraProjectionType(context.handle))) {
                 case Perspective: {
                     SharedPointer<PerspectiveProjection> perspectiveProjection = PerspectiveProjection::Create();
                     perspectiveProjection->SetAspectRatio(CFFReader::GetCameraAspect(context.handle));
                     perspectiveProjection->SetNearZ(CFFReader::GetCameraZNear(context.handle));
                     perspectiveProjection->SetFarZ(CFFReader::GetCameraZFar(context.handle));
                     perspectiveProjection->SetFovYDegrees(CFFReader::GetCameraFov(context.handle));
                     camera.SetProjection(perspectiveProjection);
                     break;
                                   }
                 case Orthographic: {
                     SharedPointer<OrthographicProjection> orthographicProjection = OrthographicProjection::Create();
                     orthographicProjection->SetNearZ(CFFReader::GetCameraZNear(context.handle));
                     orthographicProjection->SetFarZ(CFFReader::GetCameraZFar(context.handle));
                     orthographicProjection->SetWidth(CFFReader::GetCameraWidth(context.handle));
                     orthographicProjection->SetHeight(CFFReader::GetCameraHeight(context.handle));
                     camera.SetProjection(orthographicProjection);
                     break;
                                    }
                 case Generic: {
                     SharedPointer<GenericProjection> genericProjection = GenericProjection::Create();
                     const Float* projectionMatrix = 0;
                     CFFReader::GetCameraProjectionMatrix(context.handle, projectionMatrix);
                     genericProjection->SetMatrix(Matrix4(projectionMatrix));
                     camera.SetProjection(genericProjection);
                     break;
                               }
                 default: break;
            }

            Vector3 upVector;
            Vector3 lookAtVector;
            FEATSTD_LINT_CURRENT_SCOPE(864, "Violates MISRA C++ 2008 Required Rule 5-0-1: access does not depend on order")
            CFFReader::GetCameraLookatVector(context.handle, lookAtVector[0], lookAtVector[1], lookAtVector[2]);
            FEATSTD_LINT_CURRENT_SCOPE(864, "Violates MISRA C++ 2008 Required Rule 5-0-1: access does not depend on order")
            CFFReader::GetCameraUpVector(context.handle, upVector[0], upVector[1], upVector[2]);
            if (!camera.SetUpAndLookAtVectors(upVector, lookAtVector)) {
                FEATSTD_LOG_WARN("Asset loading of Up and LookAt vector failed for camera \"%s\".", camera.GetName());
            }

            if (Benchmark == CFFReader::GetCameraRenderStrategy(context.handle)) {
                BenchmarkCameraRenderStrategy* benchmarkRenderStrategy = FEATSTD_NEW(BenchmarkCameraRenderStrategy);
                if (benchmarkRenderStrategy != 0) {
                    benchmarkRenderStrategy->SetThreshold(CFFReader::GetCameraBenchmarkTreshold(context.handle));
                    camera.SetCameraRenderStrategy(benchmarkRenderStrategy);
                }
            }

            ClearMode clearMode;
            CFFReader::GetCameraClearColor(context.handle, f1, f2, f3, f4);
            clearMode.SetClearColor(Color(f1, f2, f3, f4));
            clearMode.SetColorClearEnabled(CFFReader::GetCameraIsColorClearEnabled(context.handle));
            clearMode.SetClearDepth(CFFReader::GetCameraClearDepth(context.handle));
            clearMode.SetDepthClearEnabled(CFFReader::GetCameraIsDepthClearEnabled(context.handle));
            clearMode.SetStencilClearValue(CFFReader::GetCameraStencilClearValue(context.handle));
            clearMode.SetStencilClearEnabled(CFFReader::GetCameraIsStencilClearEnabled(context.handle));
            camera.SetClearMode(clearMode);

            camera.SetCameraEffectiveAlphaEnabled(CFFReader::GetCameraApplyCameraAlpha(context.handle));

            return true;
        }

        bool AssetReaderBase<Candera::Camera>::ReadSecondPass(Camera& camera, LoaderContext& context)
        {
            const AssetId& lookAtNodeAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetCameraLookatNode(context.handle));
            if (lookAtNodeAssetId.IsValid()) {
                Node* lookAtNode = context.provider->GetNodeByAssetId(lookAtNodeAssetId);
                if (lookAtNode == 0) {
                    FEATSTD_LOG_ERROR("Failed to read lookat Node for Camera" AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }
                static_cast<void>(camera.SetLookAtNode(lookAtNode));
            }

            if (OcclusionCulling == CFFReader::GetCameraRenderStrategy(context.handle)) {
                OcclusionCullingCameraRenderStrategy* strategy = FEATSTD_NEW(OcclusionCullingCameraRenderStrategy);
                if (0 != strategy) {
                    strategy->Initialize(camera.GetScene(), static_cast<Query::Type>(CFFReader::GetCameraQueryType(context.handle)));
                    camera.SetCameraRenderStrategy(strategy);
                }
            }

            Int32 childernCount = CFFReader::GetNode3DChildrenCount(context.handle);
            Mesh* skyBoxMesh = 0;
            for (Int32 childIndex = 0; (childIndex < childernCount) && (skyBoxMesh == 0); ++childIndex) {
                const AssetDataHandle& childHandle = CFFReader::GetNode3DChildrenElementAt(context.handle, childIndex);
                if (!childHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read child for Camera" AssetIdLogStr, AssetIdLogArgs(context.id));
                    return false;
                }
                if (static_cast<AssetObjectType>(CFFReader::GetCanderaObjectItemType(childHandle)) == AotSkyBoxMesh) {
                    Candera::Internal::AssetId result = AssetIdFunctions::GetAssetId(CFFReader::GetCanderaObjectCanderaId(childHandle));
                    if (!result.IsValid()) {
                        FEATSTD_LOG_DEBUG("Mesh AssetId is not valid");
                    }
                    skyBoxMesh = Dynamic_Cast<Mesh*>(camera.GetChild(AssetIdFunctions::GetNodeId(result)));
                }
            }

            if (skyBoxMesh != 0) {
                MemoryManagement::SharedPointer<SkyBox> skyBox = SkyBox::Create();
                if (skyBox != 0) {
                    Mesh* actualMesh = skyBox->GetSkyBoxMesh();
                    skyBox->SetSkyBoxMesh(skyBoxMesh);
                    if (actualMesh != 0) {
                        // Copy vertex buffer and appearance to the existing sky box mesh
                        if (skyBoxMesh->GetVertexBuffer() == 0) {
                            skyBoxMesh->SetVertexBuffer(actualMesh->GetVertexBuffer());
                        }
                        SharedPointer<Appearance> newAppearance = skyBoxMesh->GetAppearance();
                        if (newAppearance == 0) {
                            skyBoxMesh->SetAppearance(actualMesh->GetAppearance());
                        }
                        else {
                            SharedPointer<Appearance> actualAppearance = actualMesh->GetAppearance();
                            while ((newAppearance != 0) && (actualAppearance != 0)) {
                                if (newAppearance->GetMaterial() == 0) {
                                    newAppearance->SetMaterial(actualAppearance->GetMaterial());
                                }
                                if (newAppearance->GetRenderMode() == 0) {
                                    newAppearance->SetRenderMode(actualAppearance->GetRenderMode());
                                }
                                if (newAppearance->GetShader() == 0) {
                                    newAppearance->SetShader(actualAppearance->GetShader());
                                }
                                if (newAppearance->GetShaderParamSetter() == 0) {
                                    newAppearance->SetShaderParamSetter(actualAppearance->GetShaderParamSetter());
                                }

                                for (UInt i = 0; i < CANDERA_MAX_TEXTURE_UNIT_COUNT; i++) {
                                    SharedPointer<Texture> texture = newAppearance->GetTexture(i);
                                    if (texture.PointsToNull()) {
                                        if (!newAppearance->SetTexture(actualAppearance->GetTexture(i), i)) {
                                            FEATSTD_LOG_WARN("Set Texture failed for appearance %s.", newAppearance->GetName());
                                        }
                                    }
                                }

                                newAppearance = newAppearance->GetNextPass();
                                actualAppearance = actualAppearance->GetNextPass();
                            }
                            actualMesh->Dispose();
                        }
                        if (!skyBoxMesh->GetParent()->RemoveChild(skyBoxMesh)) {
                            FEATSTD_LOG_WARN("Tree structure altered due to RemoveChild fail for Camera " AssetIdLogStr, AssetIdLogArgs(context.id));
                        }

                        ClearMode clearMode = camera.GetClearMode();
                        clearMode.SetSkyBoxEnabled(true);
                        clearMode.SetSkyBox(skyBox);
                        camera.SetClearMode(clearMode);
                    }
                }
            }

            return true;
        }

        Camera* AssetBuilderBase<Camera*>::Create(LoaderContext& /*context*/)
        {
            return Camera::Create();
        }

        void AssetBuilderBase<Camera*>::Dispose(const Camera* camera)
        {
            if (camera != 0) {
                if (camera->GetCameraRenderStrategy() != 0) {
                    FEATSTD_DELETE(camera->GetCameraRenderStrategy());
                }
            }
        }
    }
}
