//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextNode2DAssetBuilder.h"

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

#include <CanderaAssetLoader/AssetLoaderBase/CffReader/TextNode2DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

#include <Candera/Engine2D/Core/TextNodeRenderer/BitmapTextNodeRenderer.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/SurfaceTextNodeRenderer.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/GlyphTextNodeRenderer.h>
#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNode2DLayouter.h>
#endif
#ifdef CANDERA_2D_OVER_3D_ENABLED
#include <Candera/Engine3D/Core/GlyphAtlasTextNodeRenderer.h>
#endif

#include <CanderaWidget/WidgetBase/StringDataTypes.h>

#include "Effect2DAssetBuilder.h"

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;
    using namespace TextRendering;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<TextNode2D>::ReadFirstPass(TextNode2D& textNode, LoaderContext& context)
        {
            const AssetId& styleAssetId = AssetIdFunctions::GetAssetId(CFFReader::GetTextNode2DStyle(context.handle));
            if (styleAssetId.IsValid()) {
                const SharedStyle::SharedPointer& style = DefaultAssetProvider::GetInstance().GetTextStyleByAssetId(styleAssetId);
                if (style == 0) {
                    FEATSTD_LOG_WARN("Style " AssetIdLogStr " not forund for TextNode " AssetIdLogStr,
                        AssetIdLogArgs(styleAssetId), AssetIdLogArgs(context.id));
                }
                else {
                    textNode.SetStyle(style);
                }
            }
            else {
                FEATSTD_LOG_WARN("TextNode " AssetIdLogStr " has no style attached!", AssetIdLogArgs(context.id));
            }

            FeatStd::String convertedText;
            const Char* textInAsset = CFFReader::GetTextNode2DTextForAsset(context.handle);
            if (MetaInfo::DataType<FeatStd::String>::ConvertFromString(convertedText, textInAsset)) {
                textNode.SetText(convertedText);
            }
            else {
                textNode.SetText(FeatStd::String(textInAsset));
            }

            textNode.SetAsyncPreRenderEnabled(CFFReader::GetTextNode2DIsAsyncPreRenderEnabled(context.handle));

            TextNodeRenderer* textNodeRenderer = 0;
            switch (CFFReader::GetTextNode2DRenderer(context.handle)) {
                case 0: textNodeRenderer = BitmapTextNodeRenderer::Create(); break;
                case 1: textNodeRenderer = SurfaceTextNodeRenderer::Create(); break;
                case 2: textNodeRenderer = &GlyphTextNodeRenderer::GetInstance(); break;
                case 3: textNodeRenderer = &GlyphCacheTextNodeRenderer::GetInstance(); break;
                case 4: textNodeRenderer = 
#ifdef CANDERA_2D_OVER_3D_ENABLED 
                    GlyphAtlasTextNodeRenderer::Create(); break;
#else
                    BitmapTextNodeRenderer::Create(); break;
#endif
                default: FEATSTD_LOG_ERROR("Invalid renderer for TextNode " AssetIdLogStr, AssetIdLogArgs(context.id));
            }

            if (textNodeRenderer != 0) {
                textNode.SetTextNodeRenderer(textNodeRenderer);
            } 
            else {
                FEATSTD_LOG_WARN("TextNode " AssetIdLogStr " has no renderer attached!", AssetIdLogArgs(context.id));
            }

#if defined(CANDERA_LAYOUT_ENABLED)
            if (CFFReader::GetTextNode2DIsTextLayoutEnabled(context.handle)) {
                textNode.SetLayouter(&TextNode2DLayouter::GetDefault());
            }

            TextNode2DLayouter::SetMultiLineEnabled(textNode, CFFReader::GetTextNode2DIsMultiLineEnabled(context.handle));
            TextNode2DLayouter::SetWordWrapEnabled(textNode, CFFReader::GetTextNode2DIsWordWrapEnabled(context.handle));
            TextNode2DLayouter::SetTextAlignment(textNode, static_cast<TextNode2DLayouter::TextAlignment>(CFFReader::GetTextNode2DTextAlignment(context.handle)));
            TextNode2DLayouter::SetLineSpacing(textNode, CFFReader::GetTextNode2DLineSpacing(context.handle));
            TextNode2DLayouter::SetTrimming(textNode, static_cast<TextNode2DLayouter::Trimming>(CFFReader::GetTextNode2DTrimming(context.handle)));

            FeatStd::String convertedTrimmingText;
            const Char* trimmingTextInAsset = CFFReader::GetTextNode2DTrimmingTextForAsset(context.handle);
            if (MetaInfo::DataType<FeatStd::String>::ConvertFromString(convertedTrimmingText, trimmingTextInAsset)) {
                TextNode2DLayouter::SetTrimmingText(textNode, convertedTrimmingText);
            }
            else {
                TextNode2DLayouter::SetTrimmingText(textNode, trimmingTextInAsset);
            }
#endif

            return true;
        }

        TextNode2D* TextNode2DAssetBuilder::Create(LoaderContext& /*context*/)
        {
            return TextNode2D::Create();
        }
    }
}
