//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderNodeAssetBuilder.h"
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Effects/Effect2D.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/RenderNode2DCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/EffectCollectionCffReader.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaAssetLoader/AssetLoader2D/AssetBuilders/Effect2DAssetBuilder.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace MemoryManagement;

    namespace Internal {
        FEATSTD_LOG_SET_REALM(LogRealm::CanderaAssetLoader);

        bool AssetReaderBase<RenderNode>::ReadFirstPass(RenderNode& renderNode, LoaderContext& context)
        {
            renderNode.SetSnapToDevicePixelEnabled(CFFReader::GetRenderNode2DIsSnapToDevicePixelEnabled(context.handle));

            const AssetDataHandle& effectCollectionHandle = CFFReader::GetRenderNode2DEffects(context.handle);
            if (!effectCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read effect collection for RenderNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            Int32 effectCount = CFFReader::GetEffectCollectionChildrenCount(effectCollectionHandle);
            for (Int32 effectIndex = 0; effectIndex < effectCount; ++effectIndex) {
                const AssetDataHandle& effectHandle = CFFReader::GetEffectCollectionChildrenElementAt(effectCollectionHandle, effectIndex);
                if (!effectHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read effect no.%d for RenderNode " AssetIdLogStr, effectIndex, AssetIdLogArgs(context.id));
                    return false;
                }

                LoaderContext effectContext = context.Clone(effectHandle);
                SharedPointer<Effect2D> effect = AssetBuilder<SharedPointer<Effect2D> >::CreateAndBuildFirstPass(effectContext);
                if (!effect.PointsToNull()) {
                    if (!renderNode.AddEffect(effect.GetPointerToSharedInstance())) {
                        FEATSTD_LOG_WARN("Failed to add effect for RenderNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                    }
                }
                else {
                    FEATSTD_LOG_WARN("Failed to create effect for RenderNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                }
            }

            return true;
        }

        bool AssetReaderBase<RenderNode>::ReadSecondPass(RenderNode& renderNode, LoaderContext& context)
        {
            UInt8 nodeEffectIndex = 0;
            const AssetDataHandle& effectCollectionHandle = CFFReader::GetRenderNode2DEffects(context.handle);
            if (!effectCollectionHandle.IsValid()) {
                FEATSTD_LOG_ERROR("Failed to read effect collection for RenderNode " AssetIdLogStr, AssetIdLogArgs(context.id));
                return false;
            }

            Int32 effectCount = CFFReader::GetEffectCollectionChildrenCount(effectCollectionHandle);
            for (Int32 effectIndex = 0; effectIndex < effectCount; ++effectIndex) {
                const AssetDataHandle& effectHandle = CFFReader::GetEffectCollectionChildrenElementAt(effectCollectionHandle, effectIndex);
                if (!effectHandle.IsValid()) {
                    FEATSTD_LOG_ERROR("Failed to read effect no.%d for RenderNode " AssetIdLogStr, effectIndex, AssetIdLogArgs(context.id));
                    return false;
                }

                const Char* effectName = AssetProviderFunctions::GetName(context.provider, context.repositoryId, CFFReader::GetCanderaObjectCanderaName(effectHandle));
                Effect2D* effect = renderNode.GetEffect(nodeEffectIndex);

                if ((0 != effect) && (effectName == effect->GetName())) {
                    LoaderContext effectContext = context.Clone(effectHandle);
                    if (!AssetBuilder<SharedPointer<Effect2D> >::BuildSecondPass(SharedPointer<Effect2D>(effect), effectContext)) {
                        FEATSTD_LOG_WARN("Failed to load effect %s for RenderNode " AssetIdLogStr, effectName, AssetIdLogArgs(context.id));
                    }

                    nodeEffectIndex++;
                }
                else {
                    FEATSTD_LOG_WARN("Failed to find effect %s for RenderNode " AssetIdLogStr, effectName, AssetIdLogArgs(context.id));
                }
            }

            return true;
        }

        RenderNode* RenderNodeAssetBuilder::Create(LoaderContext& /*context*/)
        {
            return RenderNode::Create();
        }

    }
}
