//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextRenderer.h"
#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/Alignment.h>
#include <Candera/TextEngine/Font.h>
#include <Candera/TextEngine/TextSize.h>
#include <Candera/TextEngine/TextRenderContext.h>
#include <Candera/TextEngine/TextLayoutStrategy.h>
#include <Candera/TextEngine/Internal/Parser.h>
#include <Candera/TextEngine/Internal/Layouter.h>
#include <Candera/TextEngine/Internal/Shaper.h>
#include <Candera/TextEngine/Internal/GlyphRenderer.h>
#include <Candera/TextEngine/LayoutingOptions.h>
#include <Candera/TextEngine/ShapingOptions.h>
#include <Candera/TextEngine/TextProperties.h>
#include <Candera/TextEngine/Internal/MeasureTextRenderContext.h>
#include <Candera/TextEngine/AlignTextMeasureContext.h>
#include <Candera/TextEngine/Internal/ShaperInfo.h>
#include <Candera/TextEngine/Internal/GlyphRenderInfo.h>
#include <Candera/TextEngine/Internal/StyleTools.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>

#include <Candera/System/GlobalizationBase/CultureManager.h>
#include <Candera/System/Diagnostics/Log.h>
#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);

    using namespace Globalization;

    namespace TextRendering {
        // ----------------------------------------------------------------------------
        // Use text rendering internals.
        using namespace Internal;

#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSection* Internal::TextRenderLocks::GetGlyphLock()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_glyphLock);
            return &s_glyphLock;
        }
        static FeatStd::Internal::CriticalSection* s_forceCsInit = Internal::TextRenderLocks::GetGlyphLock();
#endif

        static bool LayoutStrategyInit(TextLayoutStrategy*& layoutStrategy, TextLayoutStrategy::Pass pass)
        {
            if (layoutStrategy != 0) {
                TextLayoutStrategy::Action action =
                    layoutStrategy->OnInit(
                    pass);
                if (action == TextLayoutStrategy::Break) {
                    return true;
                }
                else {
                    if (action == TextLayoutStrategy::Release) {
                        layoutStrategy = 0;
                    }
                }
            }
            return false;
        }

        static bool LayoutStrategyLineBreak(TextLayoutStrategy*& layoutStrategy, const Char* lineStart, TextLayoutStrategy::LineBreakType breakType)
        {
            if (layoutStrategy != 0) {
                TextLayoutStrategy::Action action =
                    layoutStrategy->OnLineBreak(
                    lineStart,
                    breakType);
                if (action == TextLayoutStrategy::Break) {
                    return true;
                }
                else {
                    if (action == TextLayoutStrategy::Release) {
                        layoutStrategy = 0;
                    }
                }
            }
            return false;
        }

        static bool LayoutStrategyGlyphSkiping(TextLayoutStrategy*& layoutStrategy, TextPosition textPosition)
        {
            bool result = false;
            if (0 != layoutStrategy) {
                TextLayoutStrategy::Action action = layoutStrategy->OnGlyphProcessing(textPosition);
                if (TextLayoutStrategy::Skip == action) {
                    result = true;
                }
                if (TextLayoutStrategy::Release == action) {
                    layoutStrategy = 0;
                }
            }
            return result;
        }

        class ProcessChunkParams {
        public:
            ProcessChunkParams() :
                m_xOffset(0),
                m_isLeadingSpacesIgnored(false),
                m_isFirstChunk(true),
                m_isFirstGlyph(true),
                m_isActualBlit(false),
                m_isLastGlyphLeftSpace(false),
                m_isLastGlyphZeroWidth(false),
                m_skippedLastGlyph(false),
                m_isXPosRealign(false),
                m_isWordWrapEnabled(false)
            {}

            inline bool IsLastGlyphSpace() const
            {
                return (m_isActualBlit == true) &&
                    (m_isLastGlyphLeftSpace == true);
            }

            inline bool IsRepositionRequired() const
            {
                return (m_isActualBlit == true) && // only actual blit should do anything to the glyphpositions
                       (m_isFirstGlyph == true) && // is it the first glyph...
                       (m_isFirstChunk == true) &&   // in the first chunk - as it only has to be corrected once
                       (m_isXPosRealign == true);
            }


            //  private:
            PixelPosition m_xOffset;
            bool m_isLeadingSpacesIgnored : 1;
            bool m_isFirstChunk : 1;
            bool m_isFirstGlyph : 1;
            bool m_isActualBlit : 1;
            bool m_isLastGlyphLeftSpace : 1;
            bool m_isLastGlyphZeroWidth : 1;
            bool m_skippedLastGlyph:1;
            bool m_isXPosRealign : 1;
            bool m_isWordWrapEnabled:1;
        };

        // ----------------------------------------------------------------------------
        static void ProcessChunk(
            TextRenderContext& context,
            GlyphRenderer& glyphRenderer,
            const Style& style,
            FontIdentifier fontIdentifier,
            const TChar* text,
            TextLength start,
            TextLength length,
            const Internal::ShaperInfo& shaperInfo,
            TextCoordinate& cursorPosition,
            PixelPosition glyphSpacing,
            TextLayoutStrategy*& layoutStrategy,
            ProcessChunkParams& chunkParams)
        {
            const TextRect& clipRect = context.GetClipRect();
            const Font& font = Internal::StyleTools::GetFontByIdentifier(style, fontIdentifier);

            // Skip rendering if font is invalid
            if (!font.IsValid()){
                return;
            }

            glyphRenderer.SetDefaultCodepoint(style.GetDefaultCodepointByIdentifier(fontIdentifier));

#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(TextRenderLocks::GetGlyphLock());
#endif
            // Create and initialize shaper.
            Shaper& shaper = Shaper::GetDefaultShaperInstance();
            if (!shaper.Initialize(
                glyphRenderer,
                font,
                text + start,
                length,
                shaperInfo)) {
                FEATSTD_LOG_WARN("Shaper initialization failed.");
            }

            const UInt8 faceId = font.GetFaceId();
            const Int16 fontHeight = font.GetHeight();
            const Int16 fontWidth = font.GetWidth();
            // Iterate through glyphs.
            while (!shaper.IsEnd()) {
                bool skipSpaces = ((chunkParams.m_isWordWrapEnabled == true) &&
                    (shaper.GetDirection() == GlyphBitmap::RightToLeft) &&
                                    (chunkParams.m_isFirstGlyph == true) &&
                                    (chunkParams.m_isFirstChunk == true) &&
                                    (chunkParams.m_isLeadingSpacesIgnored == true));

                // Get glyph index from shaper.
                GlyphIndex glyph = shaper.GetGlyph();
                bool skip = LayoutStrategyGlyphSkiping(layoutStrategy, start + shaper.GetCharacterPosition());
                if ((glyph != 0) && (glyph != 0xfffe) && (!skip)) {
                    GlyphBitmap glyphBitmap = GlyphBitmap();
                    glyphBitmap.fontIdentifier = fontIdentifier;
                    glyphBitmap.faceIdentifier.m_value = faceId;
                    glyphBitmap.fontHeight = fontHeight;
                    glyphBitmap.fontWidth = fontWidth;
                    glyphBitmap.glyphIndex = glyph;

                    // Get glyph bitmap & info from glyph renderer.
                    if (glyphRenderer.GetGlyphBitmap(glyphBitmap, font, glyph)) {

                       if (chunkParams.m_isLastGlyphLeftSpace == true) {
                           chunkParams.m_isLastGlyphLeftSpace = false;
                       }
                       if ((glyphBitmap.width > 0) || (!skipSpaces)) {
                        chunkParams.m_skippedLastGlyph = false;
                        skipSpaces = false;
                        glyphBitmap.glyphIndex = glyph;

                        glyphBitmap.direction = shaper.GetDirection();
                        glyphBitmap.fontIdentifier = fontIdentifier;
                        glyphBitmap.faceIdentifier.m_value = faceId;
                        glyphBitmap.fontHeight = fontHeight;
                        glyphBitmap.fontWidth = fontWidth;
                        glyphBitmap.characterPosition =
                            start + shaper.GetCharacterPosition();

                        // Correct bitmap info.
                        shaper.CorrectGlyphBitmap(glyphBitmap);

                        //correct starting position
                        if (chunkParams.IsRepositionRequired()) {
                            cursorPosition.TranslateX(-(cursorPosition.GetX() + glyphBitmap.left));
                        }
                        chunkParams.m_isFirstGlyph = false;
                        chunkParams.m_isFirstChunk = false;
                        // m_isLastGlyphLeftSpace has its name because the parser should give a right left space combination
                        // however, the parser does generate a right right space combination which is wrong - but has to be handled
                        // until this is fixed
                        if ((glyphBitmap.width <= 0) &&
                            (glyphBitmap.direction == GlyphBitmap::RightToLeft) &&
                            (chunkParams.m_isLastGlyphZeroWidth == false)) {
                            chunkParams.m_isLastGlyphLeftSpace = true;
                            chunkParams.m_isLastGlyphZeroWidth = true;
                        }
                        if (glyphBitmap.width > 0){
                            chunkParams.m_isLastGlyphZeroWidth = false;
                        }

                        // Set bounds of the glyph rect.
                        TextRect glyphRect(
                            TextCoordinate(
                            cursorPosition.GetX() + glyphBitmap.left,
                            cursorPosition.GetY() - glyphBitmap.top),
                            TextSize(
                            static_cast<Int16> (glyphBitmap.width),
                            static_cast<Int16> (glyphBitmap.height)));

                        // Check whether the glyph is visible.
                        if (!glyphRect.Intersect(clipRect).IsNegative()) {
                            // Blit glyph if visible.
                            context.Blit(
                                glyphRect.GetLeft(),
                                glyphRect.GetTop(),
                                glyphBitmap);
                        }
                        else
                        {
                            context.SkippedBlit(
                                glyphRect.GetLeft(),
                                glyphRect.GetTop(),
                                glyphBitmap);
                        }

                        // Update cursor position.
                        cursorPosition.Translate(glyphBitmap.xadvance, glyphBitmap.yadvance);
                        cursorPosition.TranslateX(glyphSpacing);

                        }
                       else {
                           chunkParams.m_skippedLastGlyph = true;
                       }
                       FEATSTD_UNUSED(skipSpaces); // Handles MISRA warning 438
                    }
                }
                // Advance shaper.
                shaper.Advance();
            }
        }

        // ----------------------------------------------------------------------------
        static void ProcessLine(
            TextRenderContext &context,
            GlyphRenderer& glyphRenderer,
            Layouter& layouter,
            Parser& parser,
            const ShapingOptions& shapingOptions,
            TextLayoutStrategy*& layoutStrategy,
            bool isActualBlit)
        {
            ProcessChunkParams chunkParams;
            chunkParams.m_isLeadingSpacesIgnored = (isActualBlit);
            chunkParams.m_isActualBlit = isActualBlit;
            chunkParams.m_isXPosRealign = (layouter.IsWordByWord());
            chunkParams.m_isWordWrapEnabled = layouter.IsWordWrapEnabled();
            TextCoordinate cursorPosition = layouter.GetCursorPosition();
            while (!parser.IsLineBreak()) {
                ProcessChunk(
                    context,
                    glyphRenderer,
                    *shapingOptions.GetStyle(),
                    parser.GetFontIdentifier(),
                    parser.GetText(),
                    parser.GetStart(),
                    parser.GetLength(),
                    Internal::ShaperInfo(shapingOptions, parser.GetDirection(), context.GetGlyphOrder()),
                    cursorPosition,
                    layouter.GetGlyphSpacing(),
                    layoutStrategy,
                    chunkParams);

                bool correctCursor = false;
                // Word break is marked at the end of a chunk.
                if (parser.IsWordBreak()){
                    correctCursor = true;
                }

                // Advance parser.
                parser.AdvanceToNextChunk();

                // Line break is marked at the start of a chunk.
                if ((correctCursor) && (!parser.IsLineBreak()) && (chunkParams.m_skippedLastGlyph == false)) {
                    cursorPosition.TranslateX(layouter.ComputeNextWordSpacing());
                }
            }
        }

        // ----------------------------------------------------------------------------
        static void ProcessText(
            TextRenderContext &context,
            GlyphRenderer& glyphRenderer,
            Layouter& layouter,
            ParserData& parserData,
            TextLayoutStrategy*& layoutStrategy,
            const ShapingOptions& shapingOptions)
        {
            const Style* style = shapingOptions.GetStyle().GetPointerToSharedInstance();

            // Initialize parser.
            Parser parser(*style, parserData);
            bool parserSetupSuccess = parser.Initialize(layouter.IsWordByWord());
            if (!parserSetupSuccess) {
                return;
            }

            while (!parser.IsEnd()) {
                TextLayoutStrategy::LineBreakType breakType = TextLayoutStrategy::MandatoryBreak;
                bool isWordWrap = false;

                if (layouter.IsWordWrapEnabled()) {
                    bool searchAllowedBreak = true;
                    bool searchForcedBreak = true;

                    // Find a matching point that allows line break.
                    while (searchAllowedBreak) {
                        Layouter measureLayouter(*style);
                        measureLayouter.InitializeForMeasuring(layouter);
                        Parser lineParser(parser);
                        lineParser.AdvanceLineBreak(Parser::AllowLineBreak);

                        AlignTextMeasureContext measureContext(shapingOptions.GetStyle().GetSharedInstance(), &context);
                        measureContext.SetIgnoreTrailingSpaces(layouter.IsWordWrapEnabled());
                        AlignLayoutStrategy newLayoutStrategy(measureContext, layoutStrategy);
                        TextLayoutStrategy* newLayoutStrategyPtr = &newLayoutStrategy;
                        lineParser.StartLineParsing(false);
                        ProcessLine(measureContext, glyphRenderer, measureLayouter, lineParser, shapingOptions, newLayoutStrategyPtr, false);

                        if (layouter.IsLineMatching(measureContext.GetTextRectangle())){
                            layouter.SetLineMeasurement(measureContext.GetTextRectangle(),
                                measureLayouter.GetWordBreakCount(),
                                lineParser.IsMandatoryLineBreak());
                            parser.SetLineBreak(lineParser);
                            searchForcedBreak = false;
                            if (lineParser.IsMandatoryLineBreak()) {
                                searchAllowedBreak = false;
                                breakType = TextLayoutStrategy::MandatoryBreak;
                            }
                            else {
                                breakType = TextLayoutStrategy::AutomaticBreak;
                            }
                            isWordWrap = true;
                        }
                        else {
                            searchAllowedBreak = false;
                        }
                    }
                    // If allowed line break is not possible, break at a
                    // matching point where line break is not normally allowed.
                    while (searchForcedBreak) {
                        Layouter measureLayouter(*style);
                        measureLayouter.InitializeForMeasuring(layouter);
                        Parser lineParser(parser);
                        lineParser.AdvanceLineBreak(Parser::NoLineBreak);

                        AlignTextMeasureContext measureContext(shapingOptions.GetStyle().GetSharedInstance(), &context);
                        measureContext.SetIgnoreTrailingSpaces(layouter.IsWordWrapEnabled());
                        AlignLayoutStrategy newLayoutStrategy(measureContext, layoutStrategy);
                        TextLayoutStrategy* newLayoutStrategyPtr = &newLayoutStrategy;
                        lineParser.StartLineParsing(false);
                        ProcessLine(measureContext, glyphRenderer, measureLayouter, lineParser, shapingOptions, newLayoutStrategyPtr, false);

                        if (layouter.IsLineMatching(measureContext.GetTextRectangle())){
                            layouter.SetLineMeasurement(measureContext.GetTextRectangle(), 0, false);
                            parser.SetLineBreak(lineParser);
                            isWordWrap = true;
                            breakType = TextLayoutStrategy::ForcedBreak;
                        }
                        else {
                            searchForcedBreak = false;
                        }
                    }
                }
                if (!isWordWrap) {
                    parser.AdvanceLineBreak(Parser::MandatoryLineBreak);
                    if (layouter.IsLineMeasurementNeeded()) {
                        Layouter measureLayouter(*style);
                        measureLayouter.InitializeForMeasuring(layouter);
                        Parser lineParser(parser);
                        AlignTextMeasureContext measureContext(shapingOptions.GetStyle().GetSharedInstance(), &context);
                        measureContext.SetIgnoreTrailingSpaces(layouter.IsWordWrapEnabled());
                        AlignLayoutStrategy newLayoutStrategy(measureContext, layoutStrategy);
                        TextLayoutStrategy* newLayoutStrategyPtr = &newLayoutStrategy;
                        lineParser.StartLineParsing(false);
                        ProcessLine(measureContext, glyphRenderer, measureLayouter, lineParser, shapingOptions, newLayoutStrategyPtr, false);
                        layouter.SetLineMeasurement(measureContext.GetTextRectangle(), 0, true);
                    }
                }

                parser.StartLineParsing(context.GetGlyphOrder() == TextRenderContext::RenderOrder);
                ProcessLine(context, glyphRenderer, layouter, parser, shapingOptions, layoutStrategy, true);
                parser.AdvanceToNextLine();
                layouter.AdvanceToNextLine();

                bool mustBreak = LayoutStrategyLineBreak(
                    layoutStrategy,
                    parser.GetText() + parser.GetLineStart(),
                    breakType);
                if (mustBreak) {
                    break;
                }
            }
        }



        // ----------------------------------------------------------------------------
        static bool RenderRawText(
            TextRenderContext &context,
            const LayoutingOptions& layoutingOptions,
            const ShapingOptions& shapingOptions,
            const TextProperties& textProperties)
        {
            const Style* style = shapingOptions.GetStyle().GetPointerToSharedInstance();

            if (style == 0) {
                return false;
            }

            const TChar* text = textProperties.GetTChar();
            if (text == 0) {
                return false;
            }

            Culture::SharedPointer culture = shapingOptions.GetCulture();
            if (culture.PointsToNull()) {
                culture = CultureManager::GetInstance().GetCurrentCulture();
            }

            // Initiliaze parsing data.
            ParserData parserData;
            bool parserDataSetupSuccess = parserData.Initialize(
                textProperties.GetTChar(),
                textProperties.GetTCharLength(),
                culture.GetPointerToSharedInstance(),
                layoutingOptions.IsMultilineTextEnabled(),
                layoutingOptions.IsRightToLeftDirection(),
                shapingOptions.GetBidiBaseLevel());
            if (!parserDataSetupSuccess) {
                return false;
            }

            // Initialize layouter.
            Layouter layouter(*style);
            layouter.Initialize(layoutingOptions, culture.GetPointerToSharedInstance());

            // Layout strategy.
            TextLayoutStrategy* layoutStrategy = layoutingOptions.GetLayoutStrategy();

            // Initialize glyph renderer.
            FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: only const getter, no side effect")
                GlyphRenderInfo renderInfo = {
                context.IsRenderingEnabled(),
                context.IsCachingEnabled(),
                style->GetDefaultCodepoint(),
                context.GetGlyphCacheAccess()
            };
            GlyphRenderer glyphRenderer(renderInfo);

            if (layouter.IsTextMeasurementNeeded()) {
                // Measure text, ignoring layouting.
                Layouter measureLayouter(*style);
                measureLayouter.InitializeForMeasuring(layouter);
                AlignTextMeasureContext measureContext(shapingOptions.GetStyle().GetSharedInstance(), &context);
                measureContext.SetIgnoreTrailingSpaces(layouter.IsWordWrapEnabled());
                AlignLayoutStrategy newLayoutStrategy(measureContext, layoutStrategy);
                TextLayoutStrategy* newLayoutStrategyPtr = &newLayoutStrategy;

                if (LayoutStrategyInit(layoutStrategy, TextLayoutStrategy::Measurement)) {
                    return false;
                }
                ProcessText(
                    measureContext,
                    glyphRenderer,
                    measureLayouter,
                    parserData,
                    newLayoutStrategyPtr,
                    shapingOptions);
                layouter.SetTextMeasurement(measureContext.GetTextRectangle());
            }
            layoutStrategy = layoutingOptions.GetLayoutStrategy();
            if (LayoutStrategyInit(layoutStrategy, TextLayoutStrategy::Render)) {
                return false;
            }
            ProcessText(
                context,
                glyphRenderer,
                layouter,
                parserData,
                layoutStrategy,
                shapingOptions);

            return true;
        }

        // ----------------------------------------------------------------------------
        static bool RenderShapedText(
            TextRenderContext &context,
            const LayoutingOptions& layoutingOptions,
            const ShapingOptions& shapingOptions,
            const TextProperties& textProperties)
        {
            // Layout strategy.
            TextLayoutStrategy* layoutStrategy = layoutingOptions.GetLayoutStrategy();

            TextProperties::GlyphIterator it = textProperties.GetGlyphIterator();
            if (!it.IsValid()) {
                return true;
            }

            const Style* style = shapingOptions.GetStyle().GetPointerToSharedInstance();
            if (style == 0) {
                return false;
            }

            // Initialize glyph renderer.
            FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: only const getter, no side effect")
                GlyphRenderInfo renderInfo = {
                context.IsRenderingEnabled(),
                context.IsCachingEnabled(),
                0,
                context.GetGlyphCacheAccess()
            };
            GlyphRenderer glyphRenderer(renderInfo);
            const TextCoordinate& offset = layoutingOptions.GetOffset();

            while (it.IsValid()) {
                GlyphIndex glyph = it->GetGlyphIndex();
                FontIdentifier fontIdentifier = it->GetFontIdentifier();
                TextPosition characterPosition = it->GetCharacterPosition();
                const PixelPosition2D& position = it->GetPosition();

                bool skip = LayoutStrategyGlyphSkiping(layoutStrategy, characterPosition);
                if ((glyph != 0) && (!skip)) {
                    const Font& font = StyleTools::GetFontByIdentifier(*style, fontIdentifier);
                    UInt8 faceId = font.GetFaceId();
                    const Int16 fontHeight = font.GetHeight();
                    const Int16 fontWidth = font.GetWidth();
                    GlyphBitmap glyphBitmap = GlyphBitmap();

                    // Get glyph bitmap & info from glyph renderer.
                    bool  getGlyphBitmapSuccess;
                    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                        FeatStd::Internal::CriticalSectionLocker lock(TextRenderLocks::GetGlyphLock());
#endif
                        getGlyphBitmapSuccess = glyphRenderer.GetGlyphBitmap(glyphBitmap, font, glyph);
                    }
                    if (getGlyphBitmapSuccess) {
                        glyphBitmap.glyphIndex = glyph;
                        glyphBitmap.fontIdentifier = fontIdentifier;
                        glyphBitmap.faceIdentifier.m_value = faceId;
                        glyphBitmap.fontHeight = fontHeight;
                        glyphBitmap.fontWidth = fontWidth;
                        glyphBitmap.characterPosition = characterPosition;

                        TextCoordinate glyphPosition(position.x, position.y);
                        glyphPosition.Translate(offset.GetX(), offset.GetY());
                        // Set bounds of the glyph rect.
                        TextRect glyphRect(
                            glyphPosition,
                            TextSize(
                            static_cast<Int16> (glyphBitmap.width),
                            static_cast<Int16> (glyphBitmap.height)));

                        // Blit glyph even when it is (partly) not visible. Clipping is done during rendering.
                        context.Blit(
                            glyphRect.GetLeft(),
                            glyphRect.GetTop(),
                            glyphBitmap);
                    }
                }
                ++it;
            }

            return true;
        }

        // ----------------------------------------------------------------------------
        bool TextRenderer::Render(
            TextRenderContext &context,
            const LayoutingOptions& layoutingOptions,
            const ShapingOptions& shapingOptions,
            const TextProperties& textProperties) const
        {
            if (textProperties.GetProcessingLevel() == TextProperties::Shaped) {
                return RenderShapedText(context,
                    layoutingOptions,
                    shapingOptions,
                    textProperties);
            }
            else {
                return RenderRawText(context,
                    layoutingOptions,
                    shapingOptions,
                    textProperties);
            }
        }

    }    // namespace TextRendering
}    // namespace Candera
