//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TruncationToGlyphIteratorContext.h"

#include <Candera/TextEngine/TextRenderContexts/PreprocessedGlyphData.h>
#include <Candera/TextEngine/Internal/PreprocessingContextVectorIterator.h>
#include <Candera/TextEngine/Internal/TextProcessProperties.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/TextRenderContext.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    namespace TextRendering {

        class DummyCacheContext : public TextRenderContext {
        public:
            DummyCacheContext(Internal::TextProcessProperties::TextCache * glyphCache) : m_glyphCache(glyphCache) {}

            virtual const TextRect& GetClipRect() const override
            {
                FEATSTD_SYNCED_STATIC_OBJECT(TextRect, s_rect);
                return s_rect;
            }
            virtual void Blit(Int16 /*x*/, Int16 /*y*/, const GlyphBitmap &/*glyph*/) override {}


            virtual GlyphCacheAccess* GetGlyphCacheAccess() const override
            {
                return m_glyphCache;
            }
        private:
            Internal::TextProcessProperties::TextCache * m_glyphCache;

        };

        const Candera::TextRendering::TruncationToGlyphIteratorContext::SharedPointer& TruncationToGlyphIteratorContext::GetDefault()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(TruncationToGlyphIteratorContext::SharedPointer, s_context);
            return s_context;
        }

        TruncationToGlyphIteratorContext::TruncationToGlyphIteratorContext() :
            m_truncationVector(),
            m_layoutRectangle(Candera::TextRendering::TextRect::GetMin()),
            m_textRectangle(Candera::TextRendering::TextRect::GetMin()),
            m_ascender(0),
            m_descender(0),
            m_textCache(0)
        {
            PreprocessingContext::SetGlyphOrder(Candera::TextRendering::TextRenderContext::RenderOrder);
        }


        TruncationToGlyphIteratorContext::~TruncationToGlyphIteratorContext()
        {
            Cleanup();
        }

        void TruncationToGlyphIteratorContext::Reinit(const Candera::TextRendering::SharedStyle::SharedPointer& style, TextRendering::Internal::TextProcessProperties::TextCache* textCache)
        {
            Cleanup();
            if (textCache != 0) {
                m_textCache = TEXTENGINE_TRANSIENT_NEW(DummyCacheContext)(textCache);
            }
            Reinit(style, m_textCache);
        }

        void TruncationToGlyphIteratorContext::Reinit(const Candera::TextRendering::SharedStyle::SharedPointer& style, const TextRendering::TextRenderContext* referenceContext)
        {
            SetReferenceTextRenderContext(referenceContext);
            m_ascender = ((style.PointsToNull())?0:style->GetMetrics().ascender);
            m_descender = ((style.PointsToNull())?0:style->GetMetrics().descender);
            m_layoutRectangle = Candera::TextRendering::TextRect::GetMin();
            m_textRectangle = Candera::TextRendering::TextRect::GetMin();
            m_truncationVector.Clear();
        }

        Candera::TextRendering::TextRect& TruncationToGlyphIteratorContext::GetTruncationLayoutRect()
        {
            return m_layoutRectangle;
        }

        Candera::TextRendering::TextRect& TruncationToGlyphIteratorContext::GetTruncationTextRect()
        {
            return m_textRectangle;
        }


        Candera::TextRendering::TextPosition TruncationToGlyphIteratorContext::GetGlyphCount() const
        {
            return static_cast<Candera::TextRendering::TextPosition>(m_truncationVector.Size());
        }


        bool TruncationToGlyphIteratorContext::operator==(TruncationToGlyphIteratorContext const& comp) const
        {
            bool isEqual = (m_truncationVector.Size() == comp.m_truncationVector.Size()) &&
                (m_layoutRectangle == comp.m_layoutRectangle) &&
                (m_textRectangle == comp.m_textRectangle);
            if (!isEqual) {
                return false;
            }

            for (SizeType i = 0; i < m_truncationVector.Size(); i++) {
                if (!(m_truncationVector[i] == comp.m_truncationVector[i])) {
                    return false;
                }
            }
            return true;
        }

        Candera::TextRendering::TextRect TruncationToGlyphIteratorContext::GenerateGlyphRect(PixelPosition const& x, PixelPosition const& y, GlyphBitmap const & glyph) const
        {
            return TextRect(
                x,
                (y + glyph.top - m_ascender) + 1,
                x + static_cast<PixelPosition>(glyph.width) - 1,
                y + glyph.top - m_descender);
        }

        void TruncationToGlyphIteratorContext::SetTextRectangle(PixelPosition x, PixelPosition y, PixelPosition width, PixelPosition height)
        {
            // compute bounds
            PixelPosition left = x;
            PixelPosition right = (x + static_cast<PixelPosition>(width)) - 1;
            PixelPosition top = y;
            PixelPosition bottom = (y + static_cast<PixelPosition>(height)) - 1;

            TextRect glyphRect(left, top, right, bottom);
            if (!glyphRect.IsEmpty()) {
                m_textRectangle = m_textRectangle.Union(glyphRect);
            }
        }

        void TruncationToGlyphIteratorContext::Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph)
        {
            PreprocessedGlyphData glyphData;
            glyphData.SetFontIdentifier(glyph.fontIdentifier);
            glyphData.SetGlyphIndex(glyph.glyphIndex);
            glyphData.SetCharacterPosition(glyph.characterPosition);
            glyphData.SetPosition(x, y);
            glyphData.SetXAdvance(glyph.xadvance);
            static_cast<void>(m_truncationVector.Add(glyphData));
            m_layoutRectangle = m_layoutRectangle.Union(GenerateGlyphRect(x, y, glyph));
            SetTextRectangle(x, y, glyph.width, glyph.height);
        }

        Candera::TextRendering::PreprocessingContext::ReferenceIterator* TruncationToGlyphIteratorContext::GetReferenceIterator() const
        {
            return Candera::TextRendering::Internal::PreprocessingContextVectorIterator<FeatStd::Internal::Vector<PreprocessedGlyphData>, Candera::TextRendering::PreprocessingContext::ReferenceIterator>
                (m_truncationVector.ConstBegin(), m_truncationVector.ConstEnd()).Clone();
        }


        void TruncationToGlyphIteratorContext::Cleanup()
        {
            if (m_textCache != 0) {
                TEXTENGINE_FREE(m_textCache); m_textCache = 0;
            }
        }

    }
}
