//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextToGlyphIteratorContext_h
#define Candera_TextEngine_TextToGlyphIteratorContext_h

#include <FeatStd/Base.h>
#include <Candera/Environment.h>
#include <Candera/EngineBase/Common/Alignment.h>
#include <Candera/TextEngine/TextRenderContexts/ITextLayoutStrategyDelegate.h>
#include <Candera/TextEngine/TextRenderContexts/PreprocessedGlyphData.h>
#include <Candera/TextEngine/TextRenderContexts/TruncationToGlyphIteratorContext.h>
#include <Candera/TextEngine/GlyphBitmap.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/TextRenderContexts/GlyphDataContainer.h>
#include <Candera/TextEngine/Async/TextRenderArgs.h>

namespace Candera {
    namespace TextRendering {


        /** @addtogroup CanderaTextEngine
        *  @{
        */
        /**
        *  @brief TextToGlyphIteratorContext generates the glyph container.
        *  The main idea is, that all alignment, size-handling and so on are already done in the TextEngine.
        *  The only thing which is missing is the transformation of the information in the TextEngine into a List of glyphs
        *  So the renderer simply gets the information which glyph has to printed where in the available graphical area.
        *
        *  Currently it also handles the basics of truncation, because this is not included in the TextEngine yet.
        */
        class TextToGlyphIteratorContext : public PreprocessingContext, public ILayoutStrategyDelegate {
        public:

            /**
            * Creates a text context based on its arguments
            */
            TextToGlyphIteratorContext(TextRenderArgs::SharedPointer renderArgs);

            /**
            * Creates a default text context; the default context is invalid and does not measure anything
            */
            TextToGlyphIteratorContext();

            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            // ILayoutStrategyDelegate overrides
            //////////////////////////////////////////////////////////////////////////////////////////////////////////

            /**
            * Checks if a truncation is necessary or not and trigger it
            */
            virtual void HandleNewLine(const Char* textCursor, TextLayoutStrategy::LineBreakType lineBreakType) override;

            /**
            * @return if the text processing should be stopped or not
            */
            virtual bool StopProcessing() override;

            /**
            * Informs that a new glyph will be processed now.
            * @return stops the processing right aways
            */
            virtual bool SetGlyphProcessing(TextPosition) override;

            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            // Calculation results
            //////////////////////////////////////////////////////////////////////////////////////////////////////////

            /*
            * Returns the measured text rectangle (only valid after the text has been processed).
            */
            const TextRect& GetTextRectangle() const;
            /*
            * Returns the measured text rectangle (only valid after the text has been processed).
            */
            const TextRect& GetLayoutRectangle() const;

            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            // Preprocessing Context overrides
            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            /**
            * Returns the maximum rectangle in which the text has to be blitted
            */
            virtual const TextRect& GetClipRect() const override;
        protected:
            virtual ReferenceIterator* GetReferenceIterator() const override;
            /**
            * Called by the TextEngine (Blit) to allow own measurements and handling the provided information about a glyph and its position
            */
            virtual void Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph) override;
            /**
            * Called by the TextEngine on Glyphs which would be outside of the bounding rectangle. Basically the blit function is not called on it.
            * But measurements and other functionalities do want to know which glyphs are not handled.
            */
            virtual void SkippedMeasure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph) override;

            /**
            * Handle the Measure call from TextEngine.
            */
            void MeasureInternal(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph, bool skip);

            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            // Internal area calculation functions
            //////////////////////////////////////////////////////////////////////////////////////////////////////////

            /**
            * @return is a x coord within the clipping rectangle
            */
            bool IsInXRange(PixelPosition const xStart, PixelPosition const xEnd);
            /**
            * @return is a y coord within the clipping rectangle
            */
            bool IsInYRange(PixelPosition const y);
            /**
            * Generate the rectangle of the glyph itself
            */
            TextRect GenerateGlyphRect(PixelPosition const& x, PixelPosition const& y, GlyphBitmap const & glyph) const;
            /**
            * Generates the rectangle of the text itself (not layout)
            */
            void SetTextRectangle(PixelPosition x, PixelPosition y, PixelPosition width, PixelPosition height);

            /**
            * Validates glyphData (sets all the properties of it)
            */
            void GenerateGlyphData(PreprocessedGlyphData& glyphData, TextRect const& glyphRect, PixelPosition const& x, PixelPosition const& y, const GlyphBitmap &glyph) const;

            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            // More readable glyph container access wrapper
            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            inline GlyphDataContainer& GetGlyphs()
            {
                FEATSTD_DEBUG_ASSERT(!m_textRenderArgs.PointsToNull() && m_textRenderArgs->GetGlyphContainer() != 0);
                return *(m_textRenderArgs->GetGlyphContainer());
            }

            inline const GlyphDataContainer& GetGlyphs() const
            {
                FEATSTD_DEBUG_ASSERT(!m_textRenderArgs.PointsToNull() && m_textRenderArgs->GetGlyphContainer() != 0);
                return *(m_textRenderArgs->GetGlyphContainer());
            }

            //////////////////////////////////////////////////////////////////////////////////////////////////////////
            // Truncation functions
            //////////////////////////////////////////////////////////////////////////////////////////////////////////

            /**
            * Checks if a truncation is available or not
            */
            virtual bool IsTruncationContextValid() const;

            /**
            * Algorithm which adds the truncation text, removes glyphs which have to be removed and rearrange the line if possible
            * @param stopElement is lineStart
            */
            void ReplaceGlyphsWithTruncationText(TextPosition stopElement);

            /**
            * Re-aligns a line.
            * Use case: TextAligment center/right, last visible line is not the last line in text, truncation is activated,
            * A truncation has to be added, the text in the last line has to be arranged again
            */
            void RealignCurrentLineInX(TextPosition const lineStart, TextPosition const lineEnd);

            /**
            * Shifts the text of a line in positive or negative direction
            * Use case: last visible line is not last line in text, text is eg LTR and right aligned. Truncation has to be added
            * on either the right or left side where already text can be. Shift all the text as far as possible to the other direction
            * to reduce the glyphs which has to be overridden
            */
            void TransformCurrentLineInX(TextPosition const lineStart, TextPosition const lineEnd, PixelPosition moveWidth, bool positiveDirection);

            /**
            * invalidates a glyph so the glyph is no longer printed in the end
            * Use case: truncation is 3 glyphs long but the glyph count of glyphs which are in the area of these 3 glyphs is more than 3.
            * Not overridden glyphs which should not be visible anymore, have to be removed(runtime issue) or invalidated
            */
            void MarkGlpyhsInvalid(TextPosition startPosition, TextPosition endPosition);

            /**
            * Calculate which indices have to be removed to add a truncation instead
            */
            bool GetGlyphTruncationRemovalRange(TextPosition const firstLineElement, TextPosition const lastLineElement, TextPosition& startRemoval, TextPosition& endRemoval);

            /**
            * Overrides a glyph by new values (basically recycling existing glyphs which are not valid anymore)
            */
            void ReplaceGlyphData(PreprocessedGlyphData& glyphToBeReplaced, GlyphData const& replaceData, PixelPosition x, PixelPosition y, TextPosition charPosition, TextPosition memoryPosition) const;

            /**
            * Adjusts the rectangles of used size (text/layout) if more space is needed by truncation algorithm
            */
            void IncreaseRectanglesInX(PixelPosition newTextPosition, bool isTrailingEnd);

            // Member 
        protected:
            struct LayoutBorder {
                PixelPosition leftBorder;
                PixelPosition rightBorder;
            };
            TextRenderArgs::SharedPointer m_textRenderArgs;
            TextRect m_LayoutRectangle;
            TextRect m_textRectangle;
            TextRect m_clippedRectangle;
            PixelPosition m_bottomLine;
            PixelPosition m_lastPosition;
            TextPosition m_currTextPos;
            TextPosition m_FirstLineElement;
            TextPosition m_LastFirstLineElement;
            TextPosition m_currLinePos;
            PixelPosition m_ascender;
            PixelPosition m_descender;
            PixelPosition m_previousX;
            PixelPosition m_LineX;
            PixelPosition m_LastLineX;
            FeatStd::Internal::Vector<PreprocessedGlyphData> m_glyphsInLine;
            FeatStd::Internal::Vector<PixelPosition> m_AdvancesInLine;
            bool m_stopProcessing : 1;
            bool m_isLastLineTruncated : 1;
            bool m_isGlyphProcessing : 1;
            bool m_isNewLine : 1;
            bool m_isTruncatedGlyphRTL : 1;
            bool m_isTruncationContextValid : 1;

            inline LayoutBorder GetLayoutBorder() const
            {
                LayoutBorder lb;
                lb.leftBorder = m_textRectangle.GetLeft() - m_LayoutRectangle.GetLeft();//left border
                lb.rightBorder = m_LayoutRectangle.GetRight() - m_textRectangle.GetRight();//right border
                return lb;
            }


        };
        /** @} */ // end of CanderaTextEngine

    }// namespace TextRendering
}// namespace Candera
#endif // Candera_TextEngine_TextToGlyphIteratorContext_h
