//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextToGlyphIteratorContext.h"
#include <Candera/TextEngine/TextRenderContexts/PreprocessedGlyphData.h>
#include <Candera/TextEngine/TextRenderContexts/TruncationToGlyphIteratorContext.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/TextRenderContext.h>
#include <Candera/TextEngine/TextLayoutStrategy.h>
#include <Candera/TextEngine/Internal/PreprocessingContextVectorIterator.h>
#include <Candera/TextEngine/TextRenderContext.h>
#include <Candera/TextEngine/Internal/TextProcessProperties.h>
#include <Candera/EngineBase/Common/Alignment.h>
#include <Candera/System/Mathematics/Math.h>

namespace Candera {
    namespace TextRendering {

        class CharacterPositionComparator {
        public:
            bool operator()(PreprocessedGlyphData lhs, PreprocessedGlyphData rhs) const {
                return lhs.GetCharacterPosition() < rhs.GetCharacterPosition();
            }
        };

        TextToGlyphIteratorContext::TextToGlyphIteratorContext(TextRenderArgs::SharedPointer renderArgs) :
            PreprocessingContext(renderArgs->GetCacheContext()),
            m_textRenderArgs(renderArgs),
            m_LayoutRectangle(Candera::TextRendering::TextRect::GetMin()),
            m_textRectangle(Candera::TextRendering::TextRect::GetMin()),
            m_clippedRectangle(Candera::TextRendering::TextRect::GetMax()),
            m_bottomLine(0),
            m_lastPosition(0),
            m_currTextPos(0),
            m_FirstLineElement(0),
            m_LastFirstLineElement(0),
            m_currLinePos(0),
            m_ascender((renderArgs->GetStyle().PointsToNull())?0:renderArgs->GetStyle()->GetMetrics().ascender),
            m_descender((renderArgs->GetStyle().PointsToNull())?0:renderArgs->GetStyle()->GetMetrics().descender),
            m_previousX(0),
            m_LineX(0),
            m_LastLineX(0),
            m_stopProcessing((renderArgs->GetGlyphContainer() == 0)),
            m_isLastLineTruncated(false),
            m_isGlyphProcessing(false),
            m_isNewLine(false),
            m_isTruncatedGlyphRTL(false),
            m_isTruncationContextValid(false)
        {
            if (renderArgs->GetGlyphContainer() != 0) {
                renderArgs->GetGlyphContainer()->Reserve(m_textRenderArgs->GetTextLength());
            }
            // Correct maximum clipping rectangle based on the restrictions of client area
            if (m_textRenderArgs->GetArea().GetWidth() > 0) {
                m_clippedRectangle.SetRight(m_textRenderArgs->GetArea().GetWidth());
                m_clippedRectangle.SetLeft(0);
            }
            if (m_textRenderArgs->GetArea().GetHeight() > 0) {
                m_clippedRectangle.SetBottom(m_textRenderArgs->GetArea().GetHeight());
                m_clippedRectangle.SetTop(0);
            }
            // is needed - otherwise the glyphs come in undefined order
            PreprocessingContext::SetGlyphOrder(Candera::TextRendering::TextRenderContext::RenderOrder);
        }

        TextToGlyphIteratorContext::TextToGlyphIteratorContext() :
            PreprocessingContext(0),
            m_textRenderArgs(),
            m_LayoutRectangle(Candera::TextRendering::TextRect::GetMin()),
            m_textRectangle(Candera::TextRendering::TextRect::GetMin()),
            m_clippedRectangle(Candera::TextRendering::TextRect::GetMax()),
            m_bottomLine(0),
            m_lastPosition(0),
            m_currTextPos(0),
            m_FirstLineElement(0),
            m_LastFirstLineElement(0),
            m_currLinePos(0),
            m_ascender(0),
            m_descender(0),
            m_previousX(0),
            m_LineX(0),
            m_LastLineX(0),
            m_stopProcessing(true),
            m_isLastLineTruncated(false),
            m_isGlyphProcessing(false),
            m_isNewLine(false),
            m_isTruncatedGlyphRTL(false),
            m_isTruncationContextValid(false)
        {}

        /**
        * Generates glyph rectangle for layout rectangle
        */
        Candera::TextRendering::TextRect TextToGlyphIteratorContext::GenerateGlyphRect(PixelPosition const& x, PixelPosition const& y, GlyphBitmap const & glyph) const
        {
            return TextRect(
                x - glyph.left,
                (y + glyph.top - m_ascender) + 1,
                (x - glyph.left) + glyph.xadvance,
                y + glyph.top - m_descender);
        }

        /**
        * Alters text rectangle of whole text
        */
        void TextToGlyphIteratorContext::SetTextRectangle(PixelPosition x, PixelPosition y, PixelPosition width, PixelPosition height)
        {
            // compute bounds-
            PixelPosition left = x;
            PixelPosition right = (x + static_cast<PixelPosition>(width)) - 1;
            if ((right > m_lastPosition) || (m_FirstLineElement == m_currTextPos)) {
                m_lastPosition = right;
            }
            PixelPosition top = y;
            PixelPosition bottom = (y + static_cast<PixelPosition>(height)) - 1;

            TextRect glyphRect(left, top, right, bottom);
            if (!glyphRect.IsEmpty()) {
                m_textRectangle = m_textRectangle.Union(glyphRect);
            }
        }

        /**
        * Checks whether the pixel position (The glyph) is within the x boundary
        * (Needs a forced linebreak)
        */
        bool TextToGlyphIteratorContext::IsInXRange(PixelPosition const xStart, PixelPosition const xEnd)
        {
            PixelPosition width = xEnd - xStart;
            return ((m_textRenderArgs->GetArea().GetWidth() == -1) || (m_textRenderArgs->GetArea().GetWidth() > width));
        }

        /**
        * Checks whether the pixel position (The glyph) is within the y boundary
        * (Text preprocessing can be stopped)
        */
        bool TextToGlyphIteratorContext::IsInYRange(PixelPosition const y)
        {
            return ((m_textRenderArgs->GetArea().GetHeight() == -1) || (m_textRenderArgs->GetArea().GetHeight() >= y));
        }

        /**
        * Sets the glyph values
        */
        void TextToGlyphIteratorContext::GenerateGlyphData(PreprocessedGlyphData& glyphData, TextRect const& glyphRect, PixelPosition const& x, PixelPosition const& y, const GlyphBitmap &glyph) const
        {
            FEATSTD_UNUSED(glyphRect);
            glyphData.SetFontIdentifier(glyph.fontIdentifier);
            glyphData.SetGlyphIndex(glyph.glyphIndex);
            glyphData.SetCharacterPosition(glyph.characterPosition);
            glyphData.SetPosition(x, y);
            glyphData.SetWidth(glyph.width);
            glyphData.SetXAdvance(glyph.xadvance);
            glyphData.SetMemoryPosition(m_currTextPos);
            glyphData.SetDirectionRTL(glyph.direction == GlyphBitmap::RightToLeft);
        }

        /**
        * Replaces a glyph with another
        */
        void TextToGlyphIteratorContext::ReplaceGlyphData(PreprocessedGlyphData& glyphToBeReplaced, GlyphData const& replaceData, PixelPosition x, PixelPosition y, TextPosition charPosition, TextPosition memoryPosition) const
        {
            glyphToBeReplaced.SetFontIdentifier(replaceData.GetFontIdentifier());
            glyphToBeReplaced.SetGlyphIndex(replaceData.GetGlyphIndex());
            glyphToBeReplaced.SetCharacterPosition(charPosition);
            glyphToBeReplaced.SetMemoryPosition(memoryPosition);
            glyphToBeReplaced.SetPosition(x, y);
            glyphToBeReplaced.SetXAdvance(replaceData.GetXAdvance());
            glyphToBeReplaced.SetWidth(replaceData.GetXAdvance());
        }

        /**
        * Check whether text needs a trimming or simply cuts the text at end of line
        */
        bool TextToGlyphIteratorContext::IsTruncationContextValid() const
        {
            return (((m_textRenderArgs->GetTruncationContext() != 0) && (m_textRenderArgs->GetTruncationContext()->GetIterator().IsValid())));
        }

        void TextToGlyphIteratorContext::Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph)
        {
            if ((m_textRenderArgs.PointsToNull()) || m_textRenderArgs->GetGlyphContainer() == 0) {
                m_stopProcessing = true;
                return;
            }
            if (m_stopProcessing == true) {
                return;
            }

            MeasureInternal(x, y, glyph, false);
        }

        void TextToGlyphIteratorContext::SkippedMeasure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph)
        {
            MeasureInternal(x, y, glyph, true);
        }

        void TextToGlyphIteratorContext::MeasureInternal(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph, bool skip)
        {
            // End of y area reached 
            // -> stop further processing
            // -> but except it is the first line, which might be clipped
            // -> add truncation when last line were not truncated yet and a truncation is used
            if ((y > (m_ascender - m_descender)) && (!IsInYRange(y + glyph.top - m_descender))) {
                ReplaceGlyphsWithTruncationText(m_LastFirstLineElement);
                m_stopProcessing = true;
                return;
            }

            if (m_currLinePos > 0) //set the difference in x direction to previous glyph as xAdvance on previous glyph
            {
                if (m_AdvancesInLine[static_cast<SizeType>(m_currLinePos - 1)] != 0) // previous glyph was a white space
                {
                    m_glyphsInLine[static_cast<SizeType>(m_currLinePos - 1)].SetXAdvance(m_AdvancesInLine[static_cast<SizeType>(m_currLinePos - 1)]);
                    GetGlyphs()[m_currTextPos - 1].SetXAdvance(m_AdvancesInLine[static_cast<SizeType>(m_currLinePos - 1)]);
                }
                else // all other glyphs
                {
                    PixelPosition correctedAdvance = x - m_previousX;
                    if (static_cast<bool>(m_isNewLine)) // current x is 0 again
                    {
                        correctedAdvance = (m_textRenderArgs->GetArea().GetWidth() - m_previousX) - 1;
                    }

                    if (correctedAdvance > m_glyphsInLine[static_cast<SizeType>(m_currLinePos - 1)].GetWidth() * 2) // workaround for justified text
                    {
                        correctedAdvance = m_glyphsInLine[static_cast<SizeType>(m_currLinePos - 1)].GetXAdvance();
                    }
                    m_glyphsInLine[static_cast<SizeType>(m_currLinePos - 1)].SetXAdvance(correctedAdvance);
                    GetGlyphs()[m_currTextPos - 1].SetXAdvance(correctedAdvance);
                }
            }

            // clean up data from previous line when no truncation has happened
            if (static_cast<bool>(m_isNewLine))
            {
                m_glyphsInLine.Clear();
                m_AdvancesInLine.Clear();
                m_currLinePos = 0;
                m_isNewLine = false;
                m_previousX = 0;
            }

            if (m_currLinePos == 0)
            {
                m_LastLineX = m_LastLineX > m_LineX ? m_LineX : m_LastLineX; // always the minimum left side
                m_LineX = x;
            }
            TextRect glyphRect = GenerateGlyphRect(x, y, glyph);
            //Add glyph to line container
            static_cast<void>(m_glyphsInLine.Add(PreprocessedGlyphData::GetInvalidGlyphData()));
            GenerateGlyphData(m_glyphsInLine[m_currLinePos], glyphRect, x, y, glyph);

            //Add glyph to container
            GenerateGlyphData(GetGlyphs()[m_currTextPos], glyphRect, x, y, glyph);

            // Add advances for white space glyphs to a list
            static_cast<void>(m_AdvancesInLine.Add(glyph.xadvance));
            if (glyph.width != 0) // set it to zero for non-white space glyphs
            {
                m_AdvancesInLine[m_currLinePos] = 0;
            }

            m_previousX = x; //store current x position

            if (skip)
            {
                SetTextRectangle(m_LastLineX, y, glyph.width, glyph.height);
                PixelPosition right = (x + static_cast<PixelPosition>(glyph.width)) - 1;
                if ((right > m_lastPosition) || ((m_FirstLineElement == m_currTextPos) && (right > 0))) {
                    m_lastPosition = right;
                }
            }
            else
            {
            m_LayoutRectangle = m_LayoutRectangle.Union(glyphRect);
            SetTextRectangle(x, y, glyph.width, glyph.height);
            }

            m_currTextPos++;
            m_currLinePos++;

            m_bottomLine = y + glyph.top;
            m_isLastLineTruncated = false; // last line truncation is false again when it were true (because new glyph of new line were processed now)
            m_isGlyphProcessing = false; // acknowledge that glyph were processed

        }

        void TextToGlyphIteratorContext::TransformCurrentLineInX(TextPosition lineStart, TextPosition lineEnd, PixelPosition moveWidth, bool positiveDirection)
        {
            FEATSTD_DEBUG_ASSERT(lineStart < lineEnd);
            if (lineEnd <= 0) {
                return;
            }
            if (!positiveDirection) {
                PreprocessedGlyphData& first = GetGlyphs()[lineStart];
                PixelPosition firstPosition = first.GetPosition().x;
                PixelPosition layoutBorder = m_textRectangle.GetLeft() - m_LayoutRectangle.GetLeft();
                PixelPosition minXPos = m_clippedRectangle.GetLeft() + layoutBorder;
                /// check if space between last glyph and maximum text rectangle size is big enough
                if ((minXPos + firstPosition) < moveWidth) {
                    return;
                }
                // adjust layout rectangle and textrectangle
                IncreaseRectanglesInX(firstPosition - moveWidth, false);
                // -1 to invert the move direction (see MoveInXDirection)
                moveWidth *= -1;
            }
            else {
                PreprocessedGlyphData& last = GetGlyphs()[lineEnd - 1];
                PixelPosition lastPosition = last.GetPosition().x + last.GetWidth();
                PixelPosition layoutBorder = m_LayoutRectangle.GetRight() - m_textRectangle.GetRight();
                /// check if space between last glyph and maximum text rectangle size is big enough
                if ((m_textRenderArgs->GetArea().GetWidth() - (layoutBorder + lastPosition)) < moveWidth) {
                    return;
                }
                IncreaseRectanglesInX(lastPosition + moveWidth, true);
            }
            // enough space exists - move all by moveWidth
            GetGlyphs().MoveInXDirection(lineStart, lineEnd, moveWidth);
        }

        void TextToGlyphIteratorContext::RealignCurrentLineInX(TextPosition const lineStart, TextPosition const lineEnd)
        {
            // line is empty
            if (lineStart == lineEnd) {
                return;
            }
            FEATSTD_DEBUG_ASSERT(lineEnd > 0);
            GlyphDataContainer::PixelSpan1D lineRange = GetGlyphs().GetXSpan(lineStart, lineEnd, true);
            LayoutBorder layoutBorder = GetLayoutBorder();
            PixelPosition getRightEmptySpace = (Candera::Math::Minimum<PixelPosition>(m_LayoutRectangle.GetRight(), m_clippedRectangle.GetRight())
                - lineRange.m_endPos) + 1;
            PixelPosition getLeftEmptySpace = lineRange.m_startPos - (layoutBorder.leftBorder + m_clippedRectangle.GetLeft());

            PixelPosition moveWidth = -getLeftEmptySpace;

            if (m_textRenderArgs->GetLayoutingOptionsRef().GetHorizontalAlignment() == HCenter) {
                moveWidth = (getRightEmptySpace - getLeftEmptySpace) / 2;
            }
            if (m_textRenderArgs->GetLayoutingOptionsRef().GetHorizontalAlignment() == HRight) {
                moveWidth = getRightEmptySpace;
            }

            GetGlyphs().MoveInXDirection(lineStart, lineEnd + 1, moveWidth);
            IncreaseRectanglesInX(GetGlyphs()[lineStart].GetPosition().x, false);
            IncreaseRectanglesInX(GetGlyphs()[lineEnd].GetPosition().x + GetGlyphs()[lineEnd].GetWidth(), true);
        }

        /**
        * Calculate which indices have to be removed to add a truncation instead
        */
        bool TextToGlyphIteratorContext::GetGlyphTruncationRemovalRange(TextPosition const firstLineElement, TextPosition const lastLineElement, TextPosition& startRemoval, TextPosition& endRemoval)
        {
            // should never be reached when client area is indefinite (because layout space theoretically never runs out)
            // otherwise the text is too long for indefinite layout size and is used wrongly.
            FEATSTD_DEBUG_ASSERT(m_textRenderArgs->GetArea().GetWidth() >= 0);
            if (m_textRenderArgs->GetArea().GetWidth() < 0) {
                return false;
            }

            PixelPosition indent = 0;
            m_isTruncationContextValid = IsTruncationContextValid();
            if (m_isTruncationContextValid)
            {
                indent = m_textRenderArgs->GetTruncationContext()->GetTruncationLayoutRect().GetLeft() + m_textRenderArgs->GetTruncationContext()->GetTruncationLayoutRect().GetWidth();
            }

            if (0 == m_currLinePos)
            {
                return false;
            }

            // not enough space for truncation signs
            if (m_textRenderArgs->GetArea().GetWidth() < (indent +
                Candera::Math::Maximum<PixelPosition>(m_glyphsInLine[0].GetXAdvance(), m_glyphsInLine[0].GetWidth())))
            {
                m_isTruncationContextValid = false;
                indent = 0;
            }

            if (firstLineElement == lastLineElement) // no element in line
            {
                return true;
            }

            startRemoval = 0;
            endRemoval = 0;
            PixelPosition width = 0;
            indent = (m_textRenderArgs->GetArea().GetWidth() - indent) - 1; // the remaining width without truncation width

            for (FeatStd::Internal::Vector<PreprocessedGlyphData>::Iterator it = m_glyphsInLine.Begin(); it != m_glyphsInLine.End(); ++it) {
                width += Candera::Math::Maximum<PixelPosition>((*it).GetXAdvance(), (*it).GetWidth());
                if (width > indent)
                {
                    break;
                }
                m_isTruncatedGlyphRTL = (*it).GetDirectionRTL(); // should have then the direction of the glyph where truncation is applied
                startRemoval++;
            }
            endRemoval = static_cast<TextPosition>(m_glyphsInLine.Size() - 1);
            return true;
        }

        void TextToGlyphIteratorContext::ReplaceGlyphsWithTruncationText(TextPosition stopElement)
        {
            if (m_isLastLineTruncated == true) {
                // the line is already truncated and no new line has started
                // so no additional truncation is needed
                return;
            }

            CharacterPositionComparator comparator;
            m_glyphsInLine.Sort(comparator);

            TextPosition startingIndex;
            TextPosition endingIndex;

            // startingIndex will hold the index of the glyphs in the line sorted by reading direction, which shall be truncated
            if (!GetGlyphTruncationRemovalRange(stopElement, m_currTextPos, startingIndex, endingIndex)) {
                return;
            }

            PixelPosition leftBorder = m_textRectangle.GetLeft();
            PixelPosition x = leftBorder;
            PixelPosition y = (m_bottomLine - m_ascender) + 1;

            // Set the position to insert truncation glyphs
            // Default value: the truncated line of text can be fully displayed together with the truncation glyphs
            TextPosition replacementPos = m_glyphsInLine[endingIndex].GetMemoryPosition();
            if (startingIndex < endingIndex) // truncation inside line
            {
                replacementPos = m_glyphsInLine[startingIndex].GetMemoryPosition();
            }

            TextPosition glyphsAdded = 0;
            PreprocessingContext::Iterator itor;
            if (m_isTruncationContextValid)
            {
                itor = m_textRenderArgs->GetTruncationContext()->GetIterator();

                // checks if the available text rectangle is smaller (in y) than the rectangle needed for truncation
                // and adjusts the rectangle for it
                if (m_textRenderArgs->GetTruncationContext()->GetTruncationTextRect().GetBottom() > m_bottomLine) {
                    m_bottomLine = m_textRenderArgs->GetTruncationContext()->GetTruncationTextRect().GetBottom();
                }

                m_LayoutRectangle = m_LayoutRectangle.Union(TextRect(x,
                    y,
                    x + m_textRenderArgs->GetTruncationContext()->GetTruncationLayoutRect().GetRight(),
                    m_bottomLine));
                m_textRectangle = m_textRectangle.Union(TextRect(x,
                    y + m_textRenderArgs->GetTruncationContext()->GetTruncationTextRect().GetTop(),
                    x + m_textRenderArgs->GetTruncationContext()->GetTruncationTextRect().GetRight(),
                    y + m_textRenderArgs->GetTruncationContext()->GetTruncationTextRect().GetBottom()));

                // Glyphs for the truncation text are always added
                glyphsAdded = m_textRenderArgs->GetTruncationContext()->GetGlyphCount();
                GetGlyphs().AppendInvalidGlyphs(glyphsAdded);
            }
            endingIndex += glyphsAdded;

            if ((glyphsAdded > 0) && (static_cast<bool>(m_isTruncatedGlyphRTL))) // for right-to-left glyphs need to be moved as truncation text is inserted
            {
                for (TextPosition pos = stopElement + static_cast<TextPosition>(m_glyphsInLine.Size() - 1) + glyphsAdded; pos >= glyphsAdded + replacementPos; --pos)
                {
                    PreprocessedGlyphData& replaceFrom = GetGlyphs()[pos - glyphsAdded];
                    GetGlyphs()[pos].SetCharacterPosition(replaceFrom.GetCharacterPosition() + glyphsAdded);
                    GetGlyphs()[pos].SetGlyphIndex(replaceFrom.GetGlyphIndex());
                    GetGlyphs()[pos].SetMemoryPosition(pos);
                    GetGlyphs()[pos].SetPosition(replaceFrom.GetPosition());
                    GetGlyphs()[pos].SetWidth(replaceFrom.GetWidth());
                    GetGlyphs()[pos].SetXAdvance(replaceFrom.GetXAdvance());
                }
            }

            // Set the display position for the truncation glyphs
            for (TextPosition pos = replacementPos; pos < replacementPos + glyphsAdded; ++pos)
            {
                GetGlyphs()[pos].SetMemoryPosition(pos);
            }

            // Adapt the "MemoryPosition" (which is here the display position) in line to inserted truncation glyphs
            for (TextPosition pos = 0; static_cast<SizeType>(pos) < m_glyphsInLine.Size(); pos++)
            {
                TextPosition currMemoryPos = m_glyphsInLine[pos].GetMemoryPosition();
                if (currMemoryPos >= replacementPos)
                {
                    m_glyphsInLine[pos].SetMemoryPosition(currMemoryPos + glyphsAdded);
                }
            }

            // Insert glyphs also in the line container
            for (TextPosition pos = 0; pos < glyphsAdded; pos++)
            {
                static_cast<void>(m_glyphsInLine.Insert(static_cast<SizeType>(startingIndex + pos), PreprocessedGlyphData::GetInvalidGlyphData()));
                m_glyphsInLine[static_cast<SizeType>(startingIndex + pos)].SetMemoryPosition(replacementPos + pos);
            }

            // Set the truncation glyphs
            while (itor.IsValid()) {
                PixelPosition glyphY = y + (*itor).GetPosition().y;
                TextPosition nextPos = m_glyphsInLine[startingIndex].GetMemoryPosition();

                // replace existing glyphs
                ReplaceGlyphData(GetGlyphs()[nextPos], *itor, 0, glyphY, startingIndex, nextPos);
                ++startingIndex;
                ++itor;
            }

            // remove glyphs which are still available in the list and overlap the truncation text
            FEATSTD_LINT_NEXT_EXPRESSION(838, "MISRA C++ Rule 0-1-9 - Previously assigned value to variable 'endingIndex' has not been used: endingIndex has been used for comparison and as index!");
            endingIndex = startingIndex;
            for (TextPosition j = endingIndex; j < static_cast<TextPosition>(m_glyphsInLine.Size()); j++)
            {
                GetGlyphs().InvalidateGlyph(m_glyphsInLine[j].GetMemoryPosition());
            }
            GetGlyphs().InvalidateGlyphRange(m_currTextPos + glyphsAdded, GetGlyphs().Size()); // remove glyph(s) after truncated line

            // Walk through the line and assign accumulated x position values for remaining glyphs
            for (TextPosition j = 0; j < static_cast<TextPosition>(m_glyphsInLine.Size()); j++)
            {
                PreprocessedGlyphData& glyph = GetGlyphs()[stopElement + j];
                if (!glyph.IsInvalidGlyph())
                {
                    PixelPosition glyphY = glyph.GetPosition().y;
                    glyph.SetPosition(x, glyphY);
                    x += glyph.GetXAdvance();
                    endingIndex = stopElement + j;
                }
            }

            // re-align the current line
            RealignCurrentLineInX(stopElement, endingIndex);
            // in case that the next line is out of y range -> this line is already truncated and does not need another truncation
            m_isLastLineTruncated = true;
        }

        bool TextToGlyphIteratorContext::SetGlyphProcessing(TextPosition)
        {
            if (m_stopProcessing == true) {
                return false;
            }
            m_isGlyphProcessing = true;
            return true;
        }

        void TextToGlyphIteratorContext::HandleNewLine(const Char* textCursor, TextLayoutStrategy::LineBreakType lineBreakType)
        {
            FEATSTD_UNUSED(textCursor);
            FEATSTD_UNUSED(lineBreakType);
            if ((m_stopProcessing == true) || (m_textRenderArgs.PointsToNull()) || (m_textRenderArgs->GetGlyphContainer() == 0)) {
                return;
            }
            FEATSTD_DEBUG_ASSERT(m_currTextPos >= m_FirstLineElement);

            // Handles truncation
            if (!IsInXRange(m_LineX, m_lastPosition))
            {
                ReplaceGlyphsWithTruncationText(m_FirstLineElement);
                m_stopProcessing = true;
            }
            // Reset line params
            m_isNewLine = true;
            m_LastFirstLineElement = m_FirstLineElement;
            m_FirstLineElement = m_currTextPos;
            //m_previousX = 0;
        }

        bool TextToGlyphIteratorContext::StopProcessing()
        {
            return m_stopProcessing;
        }

        void TextToGlyphIteratorContext::IncreaseRectanglesInX(PixelPosition newTextPosition, bool isTrailingEnd)
        {
            if (isTrailingEnd) {
                PixelPosition layoutBorder = m_LayoutRectangle.GetRight() - m_textRectangle.GetRight();
                m_textRectangle.SetRight(newTextPosition);
                m_LayoutRectangle.SetRight(newTextPosition + layoutBorder);
            }
            else {
                PixelPosition layoutBorder = m_textRectangle.GetLeft() - m_LayoutRectangle.GetLeft();
                m_textRectangle.SetLeft(newTextPosition);
                m_LayoutRectangle.SetLeft(newTextPosition - layoutBorder);
            }
        }


        //////////////////////////////////////////////////////////////////////////
        // Getter
        const Candera::TextRendering::TextRect& TextToGlyphIteratorContext::GetClipRect() const
        {
            return m_clippedRectangle;
        }

        const Candera::TextRendering::TextRect& TextToGlyphIteratorContext::GetTextRectangle() const
        {
            return m_textRectangle.IsNegative()?Candera::TextRendering::TextRect::GetDefault():m_textRectangle;
        }

        const Candera::TextRendering::TextRect& TextToGlyphIteratorContext::GetLayoutRectangle() const
        {
            return m_LayoutRectangle.IsNegative()?Candera::TextRendering::TextRect::GetDefault():m_LayoutRectangle;
        }

        Candera::TextRendering::PreprocessingContext::ReferenceIterator* TextToGlyphIteratorContext::GetReferenceIterator() const
        {
            if ((m_textRenderArgs.PointsToNull()) || (m_textRenderArgs->GetGlyphContainer() == 0)) {
                return 0;
            }
            return Candera::TextRendering::Internal::PreprocessingContextVectorIterator<FeatStd::Internal::Vector<PreprocessedGlyphData>, Candera::TextRendering::PreprocessingContext::ReferenceIterator>
                (GetGlyphs().ConstBegin(), GetGlyphs().ConstEnd()).Clone();
        }


    }
}
