//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_PreprocessedGlyphData_h
#define Candera_TextEngine_PreprocessedGlyphData_h


#include <FeatStd/Base.h>
#include <Candera/Environment.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/PreprocessingContext.h>

namespace Candera {
    namespace TextRendering {

        /** @addtogroup CanderaTextEngine
        *  @{
        */

        /**
        * @brief Class used for preprocessed glyph data.
        */
        class PreprocessedGlyphData : public PreprocessingContext::GlyphData {
        public:
            /**
            * Invalid Glyph has set all values to 0 (but font identifier -1)
            * In general: A glyph index of 0 is enough to invalidate a glyph
            */
            static const PreprocessedGlyphData& GetInvalidGlyphData();

            /**
            * CTor: Per default it is an invalid glyph
            */
            PreprocessedGlyphData();

            // Override functions from PreprocessingContext::GlyphData

            /**
            * @return font identifier
            */
            virtual FontIdentifier GetFontIdentifier() const;

            /**
            * @return glyph index
            */
            virtual GlyphIndex GetGlyphIndex() const;

            /**
            * @return character position
            */
            virtual TextPosition GetCharacterPosition() const;

            /**
            * @return memory position
            */
            virtual TextPosition GetMemoryPosition() const;

            /**
            * @return position
            */
            virtual PixelPosition2D GetPosition() const;

            /**
            * @return width
            */
            virtual PixelPosition GetWidth() const;

            /**
            * @return x advance
            */
            virtual PixelPosition GetXAdvance() const;

            /**
            * Retrieves the direction of a glyph (whether it is right-to-left as part of a bidirectional chunk)
            * @return true, if the glyph has a right-to-left direction
            */
            virtual bool GetDirectionRTL() const;

            /**
            * sets font identifier
            * @param fontIdentifier of the glyph font (actual visible font identifier)
            */
            void SetFontIdentifier(FontIdentifier fontIdentifier);

            /**
            * sets glyph index
            * @param glyphIndex of the glyph (actual visible glyph identifier)
            */
            void SetGlyphIndex(GlyphIndex glyphIndex);

            /**
            * sets character position
            * @param characterPosition within the text
            */
            void SetCharacterPosition(TextPosition characterPosition);

            /**
            * sets memory position
            * @param memoryPosition within the text
            */
            void SetMemoryPosition(TextPosition memoryPosition);

            /**
            * sets position
            * @param position Combination of x and y
            */
            void SetPosition(PixelPosition2D position);

            /**
            * Same as SetPosition(PixelPosition2D) but saves creation of PixelPosition2D
            * @param x position
            * @param y position
            */
            void SetPosition(PixelPosition x, PixelPosition y);

            /**
            * Increments the current glyph position by x and y.
            * To decrement the position, negate x and y
            * Mainly used to save ctor and copy calls when the position would have to be set
            * @param x position by which the glyph will be transformed
            * @param y position by which the glyph will be transformed
            */
            void IncPosition(PixelPosition x, PixelPosition y);
            /**
            * Sets the width of a glyph (How much space does the glyph use)
            * @param width
            */
            void SetWidth(PixelPosition width);

            /**
            * Sets the position where the next glyph will be written to
            * When a glyph iterator is completed, this value is irrelevant.
            * Every glyph knows its position by then.
            * Main purpose: Information holder for complexer calculations
            */
            void SetXAdvance(PixelPosition xadvance);

            /**
            * Sets the direction of a glyph (whether it is right-to-left as part of a bidirectional chunk)
            * @param isRTL if true, the glyph has a right-to-left direction
            */
            void SetDirectionRTL(bool isRTL);

            /**
            * glyph index == 0: Invalid glyph
            * The glyph index is enough for the text engine to determine an invalid glyph.
            */
            bool IsInvalidGlyph() const;

            /**
            * Invalid Glyph has set all values to 0 (but font identifier -1)
            * In general: A glyph index of 0 is enough to invalidate a glyph
            */
            void InvalidateGlyph();

            PixelPosition GetXSpacing() const;
            bool operator==(PreprocessedGlyphData const& glyph) const;
        private:
            FontIdentifier m_fontIdentifier;
            GlyphIndex m_glyphIndex;
            TextPosition m_characterPosition;
            TextPosition m_memoryPosition;
            PixelPosition2D m_position;
            PixelPosition m_width;
            PixelPosition m_xadvance;
            bool m_directionRTL;
        };
        /** @} */ // end of CanderaTextEngine

    }// namespace TextRendering
}// namespace Candera
#endif // Candera_TextEngine_PreprocessedGlyphData_h
