//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_GlyphDataContainer_h
#define Candera_TextEngine_GlyphDataContainer_h


#include <FeatStd/Base.h>
#include <FeatStd/Container/Vector.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/Environment.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/TextRenderContexts/PreprocessedGlyphData.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>

namespace Candera {
    namespace TextRendering {

        /** @addtogroup CanderaTextEngine
        *  @{
        */

        /**
        * @brief Class to visit and modify the content of GlyphDataContainer (Every glyph is visited by entry order)
        */
        class GlyphDataContainerVisitor {
        public:
            virtual ~GlyphDataContainerVisitor() {}

            /**
            * Operates on a glyph
            * @param glyph is the glyph on which an operation will be done
            */
            virtual void Visit(PreprocessedGlyphData& glyph) = 0;

        protected:
            GlyphDataContainerVisitor() {}
        };

        /**
        * @brief Class used for accessing preprocessed data (container).
        */
        class GlyphDataContainer {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(GlyphDataContainer);
            struct PixelSpan1D {
                PixelSpan1D() :m_startPos(0), m_endPos(0) {}
                PixelSpan1D(PixelPosition startPos, PixelPosition endPos) : m_startPos(startPos),m_endPos(endPos) {}
                PixelPosition m_startPos;
                PixelPosition m_endPos;
                inline PixelPosition GetWidth()const
                {
                    return (m_endPos - m_startPos);
                }
            };

            typedef FeatStd::Internal::Vector<PreprocessedGlyphData> GlyphContainer;
            GlyphDataContainer(TextPosition const reserveSize = -1);
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION()
            {
                return GlyphDataContainer::SharedPointer(TEXTENGINE_TRANSIENT_NEW(GlyphDataContainer)());
            }
            /**
            * Adds a new Glyph entry
            * The new glyph entry is set to invalid data (Glyphindex == 0 is always invalid)
            */
            void AddNew();

            /**
            * @return glyph at specified position.
            * If textPos is equal to size (so it is the element 'after' the last one)
            * an AddNew will be implicitly called
            */
            PreprocessedGlyphData& operator[](TextPosition const textPos);

            /**
            * @return glyph at specified position.
            * An AddNew will not be called (constant declaration)
            */
            PreprocessedGlyphData const& operator[](TextPosition const textPos) const;
            /**
            * Get glyph count of container
            */
            TextPosition Size()const;

            /**
            * @return begin iterator
            */
            GlyphContainer::ConstIterator ConstBegin() const;

            /**
            * @return end iterator
            */
            GlyphContainer::ConstIterator ConstEnd() const;

            /**
            * Reserves container size (to minimize reallocation calls)
            */
            void Reserve(TextPosition const reserveSize);

            /**
            * Clears/Empty container
            */
            void Reset();

            /**
            * Special iterator for TextProperties based on ConstBegin() and ConstEnd()
            */
            PreprocessingContext::Iterator GetIterator() const;

            /**
            * Special operation (visitor) on a range of entries in the GlyphContainer
            */
            void Visit(TextPosition start, TextPosition excludedEndpos, GlyphDataContainerVisitor * visitor);
            /**
            * Special operation (visitor) on all entries of the GlyphContainer
            */
            void Visit(GlyphDataContainerVisitor * visitor);


            //////////////////////////////////////////////////////////////////////////
            // Invalidation functions
            //////////////////////////////////////////////////////////////////////////

            /**
            * Adds invalid glyphs at end of list (creates new entries)
            * @param invalidCount defines the amount of invalid glyphs which will be added.
            */
            void AppendInvalidGlyphs(TextPosition const invalidCount);

            /**
            * Invalidates entries. Does not add new ones when excludedEndpos exceeds container size
            * @param start of invalidation range
            * @param excludedEndpos end of invalidation range
            */
            void InvalidateGlyphRange(TextPosition start, TextPosition excludedEndpos);

            /**
            * Invalidates a single entry. Does not add new one when pos exceeds container size
            * @param pos glyph which will be invalidated
            */
            void InvalidateGlyph(TextPosition const pos);

            //////////////////////////////////////////////////////////////////////////
            // Transformation functions
            //////////////////////////////////////////////////////////////////////////

            /**
            * Shifts all glyphs between start and excludedEndpos by x in X-Direction
            * @param start of move range
            * @param excludedEndpos of move range
            * @param x is the value of moving into positive x direction (to shift negative, x has to be negative)
            */
            void MoveInXDirection(TextPosition start, TextPosition excludedEndpos, PixelPosition x);


            /**
            * Shifts all glyphs between start and excludedEndpos by x in X-Direction and y in Y-Direction
            * @param start of move range
            * @param excludedEndpos of move range
            * @param x is the value of moving into positive x direction (to shift negative, x has to be negative)
            * @param y is the value of moving into positive y direction (to shift negative, y has to be negative)
            */
            void MoveInDirection(TextPosition start, TextPosition excludedEndpos, PixelPosition x, PixelPosition y);

            /**
            * Shifts all glyphs of the container by x in X-Direction and y in Y-Direction
            * @param x is the value of moving into positive x direction (to shift negative, x has to be negative)
            * @param y is the value of moving into positive y direction (to shift negative, y has to be negative)
            */
            void MoveInDirection(PixelPosition x, PixelPosition y);


            //////////////////////////////////////////////////////////////////////////
            // Calculation functions
            //////////////////////////////////////////////////////////////////////////

            /**
            * Calculates the x-direction span width of a glyph range so which size is actually used for all the glyphs in it
            * No guarantee that both glyphs are in same baseline and no guarantee that glyphs between start and end are somewhere
            * outside of the range
            * @param posStart is the first position
            * @param posEnd is the last position including its width
            * @param includeAdvance also adds the pixels of the calculated advance of a glyph
            * Formula: start (x) <--> end (x + width or advance)
            */
            PixelSpan1D GetXSpan(TextPosition const posStart, TextPosition const posEnd, bool includeAdvance) const;

            /**
            * Calculates if glyph at position index is (half way) out of the bounding box in x direction (right side).
            * So the glyphs right-most position (includes the advance space) is outside the bounding box limits
            * @param index of glyph in container
            * @param maxX border of the bounding box in x direction
            * @param emptyGlyphIsTrailing a glyph with width 0 is always outside the bounding box (simply ignored) when this flag is set
            * @return glyph is outside the bounding box (on right side) == true
            */
            bool IsGlyphInXTrailing(TextPosition const index, PixelPosition maxX, bool emptyGlyphIsTrailing)const;

            /**
            * Calculates if glyph is (half way) out of the bounding box in x direction (right side).
            * So the glyphs right-most position (includes the advance space) is outside the bounding box limits
            * @param glyph
            * @param maxX border of the bounding box in x direction (greater than border == outside)
            * @param emptyGlyphIsTrailing a glyph with width 0 is always outside the bounding box (simply ignored) when this flag is set
            * @return glyph is outside the bounding box (on right side) == true
            */
            bool IsGlyphInXTrailing(PreprocessedGlyphData const& glyph, PixelPosition maxX, bool emptyGlyphIsTrailing)const;

            /**
            * Calculates if glyph at position index is (half way) out of the bounding box in x direction (left side).
            * So the glyphs left-most position (x pos of glyph) is outside the bounding box limits
            * @param index of glyph in container
            * @param maxX border of the bounding box in x direction (smaller than border == outside)
            * @param emptyGlyphIsTrailing a glyph with width 0 is always outside the bounding box (simply ignored) when this flag is set
            * @return glyph is outside the bounding box (on left side) == true
            */
            bool IsGlyphInXLeading(TextPosition const index, PixelPosition maxX, bool emptyGlyphIsTrailing)const;
            /**
            * Calculates if glyph is (half way) out of the bounding box in x direction (left side).
            * So the glyphs left-most position (x pos of glyph) is outside the bounding box limits
            * @param glyph
            * @param minX border of the bounding box in x direction (smaller than border == outside)
            * @param emptyGlyphIsTrailing a glyph with width 0 is always outside the bounding box (simply ignored) when this flag is set
            * @return glyph is outside the bounding box (on left side) == true
            */
            bool IsGlyphInXLeading(PreprocessedGlyphData const& glyph, PixelPosition minX, bool emptyGlyphIsTrailing)const;

        private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(GlyphDataContainer);
            FEATSTD_SHARED_POINTER_DECLARATION();

            TextPosition GetTrailingInvalidGlyphs();

            GlyphContainer m_textVector;
        };
        /** @} */ // end of CanderaTextEngine

    }// namespace TextRendering
}// namespace Candera
#endif // Candera_TextEngine_GlyphDataContainer_h
