//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_LayoutingOptions_h)
#define Candera_LayoutingOptions_h

#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/TextCoordinate.h>
#include <Candera/TextEngine/TextSize.h>
#include <Candera/EngineBase/Common/Alignment.h>

namespace Candera {

    namespace TextRendering {
    class TextLayoutStrategy;

/** @addtogroup CanderaTextEngine
 *  @{
 */
        /**
         *  @brief This class is used for describing the layout of the text.
         *
         *  The text space is defined by the TextRenderContext. Bitmaps sent to
         *  the TextRenderContext are associated with a position computed by the
         *  text Layouter. This defines the text space as the space in which layout
         *  takes place.
         *
         *  First step in layout text is to set up a text cursor. This cursor
         *  will walk the base line, selecting the base position of the glyphs.
         *
         *  The default layout places the origin of the text space on the first
         *  cursor position in the horizontal direction and on the ascender of the
         *  first line of text in the vertical direction.
         *  The directions of the axes of the text space are from left to right
         *  and from top to bottom.
         *
         *  The layout offset defines how the text is translated. The offset
         *  is given in text space.
         */
        class LayoutingOptions {
            public:
                /**
                 *  Constructor for default layout.
                 */
                LayoutingOptions();

                /**
                 *  Constructor for layout with an offset.
                 *  @param offset Vector by which to offset the layout.
                 */
                explicit LayoutingOptions(const TextCoordinate &offset);

                /**
                 *  Constructor for layout with an offset and alignment properties.
                 *  @param horizontalAlignment Horizontal alignment value.
                 *  @param verticalAlignment Vertical alignment value.
                 *  @param offset Vector by which to offset the layout.
                 *  @param size Size of the layouted text.
                 *  @param isMultilineTextEnabled Flag specifying if text should be layouted single or multi line.
                 *  @param isWordWrapEnabled True to apply word wrapping.
                 *  @param lineSpacing Line spacing.
                 *  @param glyphSpacing Glyph spacing.
                 *  @param isRightToLeftDirection set true when the direction shall be right to left
                 *  @param isOrientationRealignmentEnabled to enable or disable this feature
                 */
                explicit LayoutingOptions(
                    HorizontalAlignment horizontalAlignment,
                    VerticalAlignment verticalAlignment,
                    const TextCoordinate &offset,
                    const TextSize& size,
                    bool isMultilineTextEnabled,
                    bool isWordWrapEnabled = false,
                    PixelSize lineSpacing = 0,
                    PixelSize glyphSpacing = 0,
                    bool isRightToLeftDirection = false,
                    bool isOrientationRealignmentEnabled = false);

                /**
                 *  Change the offset of the layout.
                 *  @param offset Vector by which to offset the layout.
                 */
                void SetOffset(const TextCoordinate &offset);

                /**
                 *  Retrieve the offset of the layout.
                 *  @return Vector by which the layout is offset.
                 */
                const TextCoordinate& GetOffset() const;

                /**
                 *  Change the horizontal alignment.
                 *  @param alignment Horizontal alignment value.
                 */
                void SetHorizontalAlignment(HorizontalAlignment alignment);

                /**
                 *  Retrieve the horizontal alignment.
                 *  @return Horizontal alignment value.
                 */
                HorizontalAlignment GetHorizontalAlignment() const;

                /**
                 *  Change the vertical alignment.
                 *  @param alignment Vertical alignment value.
                 */
                void SetVerticalAlignment(VerticalAlignment alignment);

                /**
                 *  Retrieve the vertical alignment.
                 *  @return Vertical alignment value.
                 */
                VerticalAlignment GetVerticalAlignment() const;

                /**
                 * Set the size of the layouted text.
                 * The size is used for text alignment, text truncation and word wrapping.
                 *  If a size is not specified, the size will be considered to fit the text.
                 *  @param size Size of the layouted text.
                 */
                void SetSize(const TextSize& size);

                /**
                 *  Retrieve the size of the layouted text.
                 *  @return Size of the layouted text.
                 */
                const TextSize& GetSize() const;

                /**
                 * Set the line spacing.
                 * This value is used for determining the distance between lines.
                 * If 0 is supplied, the value from the style is used.
                 *  @param lineSpacing Line spacing.
                 */
                void SetLineSpacing(PixelSize lineSpacing);

                /**
                 *  Retrieve the line spacing.
                 *  @return Line spacing.
                 */
                PixelSize GetLineSpacing() const;

                /**
                 * Set the glyph spacing.
                 * This value is used as additional offset between glyphs.
                 * If 0 is supplied, the value from the style is used.
                 *  @param glyphSpacing Glyph spacing.
                 */
                void SetGlyphSpacing(PixelPosition glyphSpacing);

                /**
                 *  Retrieve the glyph spacing.
                 *  @return Glyph spacing.
                 */
                PixelPosition GetGlyphSpacing() const;

                /**
                 * Set multi line text layout.
                 * @param isMultilineTextEnabled Flag specifying if text should be layouted single or multi line.
                 */
                void SetMultilineTextEnabled(bool isMultilineTextEnabled);

                /**
                 * Get multi line text layout.
                 * @return  Flag specifying if text should be layouted single or multi line.
                 */
                bool IsMultilineTextEnabled() const;

                /**
                 * Enables/disables word wrapping when layout text.
                 * @param isWordWrapEnabled True to apply word wrapping.
                 */
                void SetWordWrapEnabled(bool isWordWrapEnabled);

                /**
                 * Verify whether word wrapping is enabled.
                 * @return True if word wrapping is enabled.
                 */
                bool IsWordWrapEnabled() const;

                /**
                 *  Set a layout strategy. The lifetime if the strategy needs
                 *  to be handle by the caller. Set a 0 strategy to disable it.
                 *  The default strategy is 0.
                 *  @param strategy Strategy object used to interact with the
                 *      text layout algorithm.
                 */
                void SetLayoutStrategy(TextLayoutStrategy* strategy);

                /**
                 *  Retrieve the layout strategy. Return value is non-constant
                 *  because the strategy may hold internal state.
                 *  @return Object set by SetLayoutStrategy, or 0.
                 */
                TextLayoutStrategy* GetLayoutStrategy() const;

                /**
                * Additional layout information if the main direction is right to left or not.
                * This has impact on possible alignment.
                * For text orientation the culture and font information is used.
                * @return true when the direction shall be right to left.
                */
                bool IsRightToLeftDirection() const;

                /**
                * Additional layout information if the main direction is right to left or not.
                * This has impact on possible alignment.
                * For text orientation the culture and font information is used.
                * @param isRightToLeftDirection set true when the direction shall be right to left.
                */
                void SetRightToLeftDirection(bool isRightToLeftDirection);

                /**
                * Flag to determine whether the text will be realigned based on its orientation when
                * the text is outside of its maximum bounding box.
                * E.g.:
                * Flag is enabled:  A centered line of text will be left/right aligned if the line is longer than the provided space.
                * Flag is disabled: A centered line of text will be kept centered (left and right crop) if the line is longer than the provided space.
                * @return realignment is enabled
                */
                bool IsOrientationRealignmentEnabled() const;
                /**
                * Flag to determine whether the text will be realigned based on its orientation when
                * the text is outside of its maximum bounding box.
                * E.g.:
                * Flag is enabled:  A centered line of text will be left/right aligned if the line is longer than the provided space.
                * Flag is disabled: A centered line of text will be kept centered (left and right crop) if the line is longer than the provided space.
                * @param isOrientationRealignmentEnabled to enable or disable this feature
                */
                void SetOrientationRealignmentEnabled(bool isOrientationRealignmentEnabled);

                /**
                * Flag to determine whether the TextEngine has to handle the culture direction or not.
                * Not handling is required when it is already handled by candera layout.
                * @return culture handling is required
                */
                bool IsCultureHandlingRequired() const;
                /**
                * Flag to determine whether the TextEngine has to handle the culture direction or not.
                * Not handling is required when it is already handled by candera layout.
                * @param isCultureHandlingRequired to enable or disable this feature
                */
                void SetCultureHandlingRequired(bool isCultureHandlingRequired);

            private:
                bool m_isMultilineTextEnabled : 1;
                bool m_isWordWrapEnabled : 1;
                bool m_isRightToLeftDirection : 1;
                bool m_isOrientationRealignmentEnabled:1;
                bool m_isCultureHandlingRequired:1;

                HorizontalAlignment m_horizontalAlignment;
                VerticalAlignment m_verticalAlignment;
                PixelSize m_lineSpacing;
                PixelPosition m_glyphSpacing;
                TextCoordinate m_offset;
                TextSize m_size;

                TextLayoutStrategy *m_strategy;
        };

 /** @} */ // end of CanderaTextEngine

    }    // namespace TextRendering
}    // namespace Candera

#endif    // Candera_LayoutingOptions_h
