//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextProcessResult_h
#define Candera_TextEngine_TextProcessResult_h

#include <Candera/System/MemoryManagement/SharedPointer.h>

#include <Candera/TextEngine/TextRect.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/TextRenderContexts/GlyphDataContainer.h>
#include <Candera/TextEngine/Types.h>
namespace Candera {
    namespace TextRendering {
        namespace Internal {
            /**
            * @brief TextProcessResult holds all results after a text is processed.
            * It holds the glyph iterator, the bounding box required to display the text and an offset.
            * It also has further information required or useful to handle the text result.
            */
            class TextProcessResult {
            public:
                FEATSTD_TYPEDEF_SHARED_POINTER(TextProcessResult);

                FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

                /**
                * Represents the glyph container which contains the iterator to iterate over the glyphs.
                * @return the glyph container
                */
                Candera::TextRendering::GlyphDataContainer::SharedPointer GetResultContainer() const;

                /**
                * Represents the glyph container which contains the iterator to iterate over the glyphs.
                * @param the glyph container
                */
                void SetResultContainer(Candera::TextRendering::GlyphDataContainer::SharedPointer& val);

                /**
                * @return the text rectangle calculated by processing the text.
                * The text rectangle defines the area which is occupied by text.
                */
                Candera::TextRendering::TextRect GetResultTextRectangle() const;


                /**
                * @param defines the text rectangle calculated by processing the text.
                * The text rectangle defines the area which is occupied by text.
                */
                void SetResultTextRectangle(Candera::TextRendering::TextRect const& val);

                /**
                * @return the layout rectangle calculated by processing the text.
                * The layout rectangle does not represent the area used by the text but the layout area in which
                * the text should be seen.
                */
                Candera::TextRendering::TextRect GetResultLayoutRectangle() const;

                /**
                * @param defines the layout rectangle calculated by processing the text.
                * The layout rectangle does not represent the area used by the text but the layout area in which
                * the text should be seen.
                */
                void SetResultLayoutRectangle(Candera::TextRendering::TextRect const& val);

                /**
                * @return style with which the resulting glyph container were generated
                */
                Candera::TextRendering::SharedStyle::SharedPointer GetStyle() const;

                /**
                * @param style with which the resulting glyph container were generated
                */
                void SetStyle(Candera::TextRendering::SharedStyle::SharedPointer& val);

                /**
                * The text position can receive an offset to move the text in a direction. 
                * It is an additive offset to each x and y coordinate.
                * @return offset of the text.
                */
                Candera::TextRendering::TextCoordinate GetOffset() const;

                /**
                * The text position can receive an offset to move the text in a direction.
                * It is an additive offset to each x and y coordinate.
                * @param offset of the text.
                */
                void SetOffset(Candera::TextRendering::TextCoordinate const& val);

                /**
                * Result analyzed is used to mark the result as already consumed.
                * This result can be consumed by e.g. TextProcessHelper::RenderText. 
                * It can be used as indicator that it is not required to render the text anymore,
                * as it is already rendered.
                * @return whether the result is consumed flag is set
                */
                bool IsResultConsumed() const;
                /**
                * Result analyzed is used to mark the result as already consumed.
                * This result can be consumed by e.g. TextProcessHelper::RenderText.
                * It can be used as indicator that it is not required to render the text anymore,
                * as it is already rendered.
                * @param sets the result is consumed flag
                */
                void SetResultConsumed(bool val);
            private:
                TextProcessResult();
                GlyphDataContainer::SharedPointer m_resultContainer;
                SharedStyle::SharedPointer m_resultStyle;
                TextRect m_resultLayoutRectangle;
                TextRect m_resultTextRectangle;
                TextCoordinate m_offset;
                bool m_resultAnalyzed;
                CANDERA_SHARED_POINTER_DECLARATION();
            };
        }
    }
}
#endif // Candera_TextEngine_TextProcessResult_h
