//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextProcessProperties_h
#define Candera_TextEngine_TextProcessProperties_h


#include <FeatStd/Util/String.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Candera/System/GlobalizationBase/Culture.h>

#include <Candera/TextEngine/TextRect.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/Types.h>


namespace Candera {
    namespace Globalization {
        class CultureChangeListener;
    }
    namespace TextRendering {
        class GlyphCacheAccess;
        namespace Internal {
            struct TruncationMode {
                enum Enum {
                    None, ///< Text is not truncated. There is no replacement text given.
                    Ellipsis, ///< Text is truncated by using an ellipsis '...'.
                    CustomText ///< Text is truncated by using a custom text.
                };
            };

            struct HorizontalTextAlignment {
                enum Enum {
                    Auto,      ///< TextAlignment inhierits layout alignment.
                    Left,      ///< Text is aligned along the left margin.
                    Center,    ///< Text is not aligned, but an equal amount of padding is added relative to the right and left margin.
                    Right,     ///< Text is aligned along the right margin.
                    Justified  ///< Text is left aligned, with word spacing increased in a way that the text fills the entire available width for non end of paragraph lines.
                };
            };

            struct TextProcessInvalidation {
                enum Enum {
                    InvalidAll, ///< Invalidates the text. It has to step through all processing steps to get a valid result.
                    ValidAll ///< Validates the text. It stepped through all processing steps and the text is fully processed. No further actions required.
                };
            };
            class ITextProcessUser;
            class TextProcessPropertiesCultureChangeListener;
            /**
            * @brief TextProcessProperties is a class which holds all properties required to process a text with the TextEngine.
            */
            class TextProcessProperties {
                friend class TextProcessPropertiesCultureChangeListener;
            public:
                typedef GlyphCacheAccess TextCache;
                TextProcessProperties();
                TextProcessProperties(TextProcessProperties const& properties);
                virtual ~TextProcessProperties();

                /**
                * The class contains complex properties which are not directly linked to this class.
                * Therefore, changing a property within such a complex property cannot be detected actively.
                * These complex properties have to provide a solution to check the 'property changed' state.
                * This state can be checked by calling this method and triggers the 'active' way to invalidate a property. 
                * So that it will be recognized by the attached ITextProcessUser.
                */
                void CheckPassivePropertiesInvalidation();

                /**
                * Get text which shall be processed/rendered.
                * @return text which will be processed/rendered.
                */
                FeatStd::String GetText() const;

                /**
                * Set text to be processed/rendered.
                * @param text to be processed/rendered.
                */
                void SetText(FeatStd::String const& val);

                /**
                * Get style used for rendering.
                */
                Candera::TextRendering::SharedStyle::SharedPointer GetStyle() const;

                /**
                * Set style to be used for rendering.
                */
                void SetStyle(Candera::TextRendering::SharedStyle::SharedPointer val);

                /**
                * Get rendered truncation text.
                * @return If truncation mode is Ellipsis, an ellipsis is returned.
                * If truncation is set to none / without replacement text, an empty string is returned.
                * Otherwise the set truncation text is returned.
                */
                FeatStd::String GetSelectedTruncationText() const;

                /**
                * Get truncation text which is set by SetTruncationText.
                */
                FeatStd::String GetTruncationText() const;

                /**
                * Sets truncation text.
                * The text will be ignored when truncation mode is set to none or ellipsis.
                */
                void SetTruncationText(FeatStd::String const& val);
               
                /**
                * Get currently active truncation mode.
                */
                TruncationMode::Enum GetTruncationMode() const;

                /**
                * Set active truncation mode.
                */
                void SetTruncationMode(TruncationMode::Enum val);

                /**
                * Text Clipping defines a text which will be clipped when it is outside of the maximum defined area.
                * This happens when the text is longer than the provided space. If the clip is not enabled, nothing will be clipped.
                * @return true if clipping is enabled otherwise false
                */
                bool IsClipEnabled() const;

                /**
                * Text Clipping defines a text which will be clipped when it is outside of the maximum defined area.
                * This happens when the text is longer than the provided space. If the clip is not enabled, nothing will be clipped.
                * @param val enables/disables clipping
                */
                void SetClipEnabled(bool val);

                /**
                * This flag switches between two line spacing modes.
                * First mode (true): The line spacing is an absolute pixel value (default).
                * Second mode (false): The line spacing is a ratio of line spacing to font height.
                * @return the current mode of the two modes described above.
                */
                bool IsPixelwiseLineSpacing() const;

                /**
                * This flag switches between two line spacing modes.
                * First mode (true): The line spacing is an absolute pixel value (default).
                * Second mode (false): The line spacing is a ratio of line spacing to font height.
                * @param val sets the current mode of the two modes described above.
                */
                void SetPixelwiseLineSpacing(bool val);

                /**
                * If word wrapping is enabled then the lines that exceed the client area will be split
                * during arrangement into multiple lines. The lines will be split at word boundaries.
                * @return The word wrapping configuration.
                */
                bool IsWordWrapEnabled() const;

                /**
                * Sets the word wrapping configuration.
                * If word wrapping is enabled then the lines that exceed the client area will be split
                * during arrangement into multiple lines. The lines will be split at word boundaries.
                * @param enabled    The word wrapping enabled flag (disabled by default).
                */
                void SetWordWrapEnabled(bool val);

                /**
                * If multi line is disabled then the new line control characters will be ignored and text
                * will always be rendered in one line.
                * @return The multi line configuration.
                */
                bool IsMultiLineEnabled() const;

                /**
                * Sets the multi line configuration.
                * If multi line is disabled then the new line control characters will be ignored and text
                * will always be rendered in one line.
                * @param enabled    The multi line enabled flag (enabled by default).
                */
                void SetMultiLineEnabled(bool val);

                /**
                * The line spacing is a metric usually defined by the Font configured for text rendering. To use
                * the default value of the Font, set 0 as the line spacing property. Any other positive value
                * will override the default value with an object space fixed height (independent on the font height).
                * @return The line spacing configured.
                */
                Candera::TextRendering::PixelSize GetLineSpacing() const;

                /**
                * Sets the line spacing value.
                * The line spacing is a metric usually defined by the Font configured for text rendering. To use
                * the default value of the Font, set 0 as the line spacing property. Any other positive value
                * will override the default value with an object space fixed height (independent on the font height).
                * @param lineSpacing  Line spacing value.
                */
                void SetLineSpacing(Candera::TextRendering::PixelSize val);

                /**
                * The TextAlignment type, combined with the culture text direction, is used to
                * define how text will be horizontally aligned within its layout area.
                * @return The TextAlignment configured.
                */
                HorizontalTextAlignment::Enum GetHorizontalTextAlignment() const;

                /**
                * Sets the TextAlignment type.
                * The TextAlignment type, combined with the culture text direction, is used to
                * define how text will be horizontally aligned within its layout area.
                * @param textAlignment    The TextAlignment value.
                * @see TextAlignment
                */
                void SetHorizontalTextAlignment(HorizontalTextAlignment::Enum val);

                /**
                * The outer size restriction is a restriction for the text from outside.
                * E.g. the layout process defines an additional restriction and can be set here.
                * @return outer size restriction
                */
                Candera::TextRendering::TextSize GetOuterSizeRestriction() const;

                /**
                * The outer size restriction is a restriction for the text from outside.
                * E.g. the layout process defines an additional restriction and can be set here.
                * @param outer size restriction
                */
                void SetOuterSizeRestriction(Candera::TextRendering::TextSize val);
                
                /**
                * The inner size restriction is the actual size restriction for the text itself.
                * @return inner size restriction
                */
                Candera::TextRendering::TextSize GetInnerSizeRestriction() const;

                /**
                * The inner size restriction is the actual size restriction for the text itself.
                * @param val is the inner size restriction
                */
                void SetInnerSizeRestriction(Candera::TextRendering::TextSize val);

                /**
                * Returns the active culture. If no culture is explicitly set by SetCulture,
                * the currently active culture will be returned as culture.
                * @return the custom culture if set else the active culture
                */
                Candera::Globalization::Culture::SharedPointer GetCulture() const;

                /**
                * Allows to explicitly set a culture. If no culture is set (empty shared pointer),
                * the currently active culture will be used as default culture.
                * @param val sets the custom culture.
                */
                void SetCulture(Candera::Globalization::Culture::SharedPointer val);

                /**
                * The glyph spacing defines the additional space given between each glyphs.
                * @return the current glyph spacing.
                */
                Candera::TextRendering::PixelSize GetGlyphSpacing() const;

                /**
                * The glyph spacing defines the additional space given between each glyphs.
                * @param val sets the current glyph spacing.
                */
                void SetGlyphSpacing(Candera::TextRendering::PixelSize val);

                /**
                * The right to left direction has impact on the interpretation of the horizontal alignment.
                * If this direction is right to left, the horizontal alignment will be inverted.
                * @return true when right to left direction is set.
                */
                bool IsRightToLeftLayoutDirection() const;

                /**
                * The right to left direction has impact on the interpretation of the horizontal alignment.
                * If this direction is right to left, the horizontal alignment will be inverted.
                * @param val sets the right to left direction active when true.
                */
                void SetRightToLeftLayoutDirection(bool val);

                /**
                * The text process user which receives the information when a property has changed.
                * It also provides information of the glyph cache which shall be used.
                * This is not required to process text with a TextProcessProperties object. 
                * @return the attached invalidation handler
                */
                Candera::TextRendering::Internal::ITextProcessUser* GetTextProcessUser() const;

                /**
                * Attaches the text process user which receives the information when a property has changed.
                * It also provides information of the glyph cache which shall be used.
                * This is not required to process text with a TextProcessProperties object.
                * @param user which shall be attached.
                */
                void SetTextProcessUser(Candera::TextRendering::Internal::ITextProcessUser* val);

                /**
                * Represents the object which caches the glyph/text information.
                * @return text cache
                */
                TextCache * GetTextCache() const;

            private:
                void InvalidateProperties(TextProcessInvalidation::Enum invalidationType);

                // General
                FeatStd::String m_text;
                TextRendering::SharedStyle::SharedPointer m_style; 
                Candera::Globalization::Culture::SharedPointer m_culture;
                ITextProcessUser* m_textProcessUser;
                Candera::Globalization::CultureChangeListener * m_cultureChangeListener;

                // Dimensions
                TextSize m_outerSizeRestriction;
                TextSize m_innerSizeRestriction;

                // Truncation
                FeatStd::String m_truncationText;
                TruncationMode::Enum m_truncationMode; //<< TextNode2DLayouter::Trimming

                // Layout
                HorizontalTextAlignment::Enum m_textAlignment;
                // TODO: Vertical alignment
                PixelSize m_lineSpacing;
                PixelSize m_glyphSpacing;
                // General
                UInt16 m_styleVersion; ///< Version of style - needed to recognize style changes

                // Layout
                bool m_pixelwiseLineSpacing:1;
                bool m_wordWrapEnabled:1;
                bool m_multiLineEnabled:1;
                bool m_isRightToLeftLayoutDirection:1;

                // Truncation
                bool m_clipEnabled:1;
            };

            class SharedTextProcessProperties : TextProcessProperties {
            public:
                FEATSTD_TYPEDEF_SHARED_POINTER(SharedTextProcessProperties);

                FEATSTD_SHARED_POINTER_CREATE_DECLARATION();
            private:
                CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, Use SharedTextProcessProperties::Create)
                    SharedTextProcessProperties() {}

                FEATSTD_MAKE_CLASS_UNCOPYABLE(SharedTextProcessProperties);

                FEATSTD_LINT_NEXT_EXPRESSION(1511, "false positive - SharedStyle::SharedPointer is also used in base class")
                FEATSTD_SHARED_POINTER_DECLARATION();
            };


            /**
            * Interface for a user of the class TextrProcessProperties.
            * It provides an interface to inavalidate the calculated result based on the properties.
            * This happens when a property has changed.
            * Additionally, it provides the glyph/text information cache for the text.
            */
            class ITextProcessUser {
            public:
                virtual ~ITextProcessUser() {}

                /**
                * Represents the object which caches the glyph/text information.
                * @return text cache
                */
                virtual TextProcessProperties::TextCache * GetGlyphCache() = 0;
            protected:

                /**
                * Callback function which will be triggered as soon as a property has changed.
                */
                virtual void InvalidateProperties(TextProcessInvalidation::Enum invalidationType) = 0;
                ITextProcessUser() {}
                friend class TextProcessProperties;
            };
        }
    }
}

#endif // Candera_TextEngine_TextProcessProperties_h

