//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SimplifiedShaper.h"
#include <Candera/TextEngine/Font.h>
#include <Candera/TextEngine/Internal/GlyphRenderer.h>
#include <Candera/TextEngine/Internal/ShaperInfo.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <Candera/TextEngine/TextRenderer.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera { namespace TextRendering { namespace Internal { 

SimplifiedShaper::SimplifiedShaper() :
    m_order(TextRenderContext::RenderOrder),
    m_reorderedDataEnd(0),
    m_reorderedData(0),
    m_previousGlyph(),
    m_currentGlyph(),
    m_nextGlyph(),
    m_glyphRenderer(0),
    m_font(0)
{
}

SimplifiedShaper::~SimplifiedShaper()
{
    ReleaseReorderedData();
    m_glyphRenderer = 0;
    m_font = 0;
}


void SimplifiedShaper::CleanUp()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(Candera::TextRendering::Internal::TextRenderLocks::GetGlyphLock());
#endif
    ReleaseReorderedData();
}

bool SimplifiedShaper::Initialize(const GlyphRenderer& glyphRenderer, const Font &font, const TChar* text, TextLength textSize, const ShaperInfo& info)
{
    ReleaseReorderedData();
    if (!InitializeUnicode(text, textSize, info)) {
        return false;
    }

    m_glyphRenderer = &glyphRenderer;
    m_font = &font;
    
    m_order = info.GetGlyphOrder();

    UpdateNextIndex();
    Advance();

    if (!IsInCorrectOrder()) {
        UInt32 count = 0;
        while (!IsEnd()) {
            ++count;
            Advance();
        }

        if (count == 0) {
            return true;
        }

        GlyphData* reorderedData = TEXTENGINE_TRANSIENT_NEW_ARRAY(GlyphData, count);
        if (reorderedData == 0) {
            return false;
        }
        if (!InitializeUnicode(text, textSize, info)) {
            TEXTENGINE_DELETE_ARRAY(reorderedData);
            return false;
        }

        UpdateNextIndex();
        Advance();

        m_reorderedDataEnd = 0;
        while ((count > 0) && (!IsEnd())) {
            --count;
            reorderedData[m_reorderedDataEnd++] = m_currentGlyph;
            Advance();
        }
        if ((0 != count) || (!IsEnd())) {
            TEXTENGINE_DELETE_ARRAY(reorderedData);
            m_reorderedDataEnd = 0;
            return false;
        }

        m_reorderedData = reorderedData;

        UpdateNextIndex();
        Advance();

    }

    return true;
}

GlyphIndex SimplifiedShaper::GetGlyph() const
{
    return m_currentGlyph.glyphIndex;
}

TextPosition SimplifiedShaper::GetCharacterPosition() const
{
    return m_currentGlyph.characterPosition;
}

void SimplifiedShaper::CorrectGlyphBitmap(GlyphBitmap& glyph) const
{
    const GlyphIndex nextGlyphIndex = ((glyph.direction == GlyphBitmap::LeftToRight) ? m_nextGlyph.glyphIndex : m_previousGlyph.glyphIndex);
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::TextRendering::Internal::SimplifiedShaper::m_font, CANDERA_LINT_REASON_OPERANDNOTNULL)
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::TextRendering::Internal::SimplifiedShaper::m_glyphRenderer, CANDERA_LINT_REASON_OPERANDNOTNULL)
    PixelPosition2D kerning = m_glyphRenderer->GetGlyphKerning(m_currentGlyph.glyphIndex, nextGlyphIndex, *m_font);
    glyph.xadvance += kerning.x;
    glyph.yadvance += kerning.y;
}

void SimplifiedShaper::Advance()
{
    m_previousGlyph = m_currentGlyph;
    m_currentGlyph = m_nextGlyph;
    UpdateNextIndex();
}

bool SimplifiedShaper::IsEnd() const
{
    if (m_reorderedData != 0) {
        return (m_reorderedDataEnd == 0) && (m_currentGlyph.glyphIndex == 0);
    }
    else {
        return IsEndUnicode() && (m_currentGlyph.glyphIndex == 0);
    }
}

void SimplifiedShaper::UpdateNextIndex() 
{
    if (m_reorderedData != 0) {
        if (m_reorderedDataEnd < 1) {
            m_nextGlyph = GlyphData();
        }
        else {
            m_nextGlyph = m_reorderedData[--m_reorderedDataEnd];
        }
    }
    else {
        if (IsEndUnicode()) {
            m_nextGlyph = GlyphData();
        }
        else {
            UnicodeData data = GetUnicode();
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::TextRendering::Internal::SimplifiedShaper::m_font, CANDERA_LINT_REASON_OPERANDNOTNULL)
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(613, Candera::TextRendering::Internal::SimplifiedShaper::m_glyphRenderer, CANDERA_LINT_REASON_OPERANDNOTNULL)
            m_nextGlyph.glyphIndex = m_glyphRenderer->GetGlyphIndex(*m_font, data.codePoint);
            m_nextGlyph.characterPosition = data.characterPosition;
            AdvanceUnicode();
        }
    }
}

void SimplifiedShaper::ReleaseReorderedData()
{
    if(m_reorderedData != 0){
        TEXTENGINE_DELETE_ARRAY(m_reorderedData);
        m_reorderedData = 0;
        m_reorderedDataEnd = 0;
    }
}

}}} // namespace
