//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_Shaper_h)
#define Candera_Shaper_h
#include <Candera/TextEngine/ShapingOptions.h>

namespace Candera {
    namespace TextRendering {
        class Font;
        namespace Internal {
            class GlyphRenderer;
            class ShaperInfo;

            /** @addtogroup CanderaTextEngine
             *  @{
             */

            /**
             *  @brief Shaper is responsible to supply glyph positions and code points to Candera
             *  text rendering function.
             *  This is a Candera private class and must not be used by other components
             */
            class Shaper {
            public:
                virtual ~Shaper();
                /**
                 *  Initialize the shaper.
                 *  @param font Font from which the glyph information should be retrieved.
                 *  @param text Text on which shaping is applied.
                 *  @param textSize Number of characters to be processed.
                 *  @param info Parameters used to configure the shaper.
                 *  @return True if the initialization is successful.
                 */
                virtual bool Initialize(const GlyphRenderer& glyphRenderer, const Font &font, const TChar* text, TextLength textSize, const ShaperInfo& info) = 0;

                /**
                 *  Retrieve the index within the font of the current glyph.
                 *  @return The glyph index.
                 */
                virtual GlyphIndex GetGlyph() const = 0;

                /**
                 *  Retrieve the direction of the current glyph.
                 *  @return The glyph direction.
                 */
                virtual GlyphBitmap::Direction GetDirection() const = 0;

                /**
                 *  Retrieve the position within the original text of the current glyph.
                 *  @return The character position.
                 */
                virtual TextPosition GetCharacterPosition() const = 0;

                /**
                 *  Apply shaping corrections of the current glyph to the bitmap.
                 *  @param glyph The glyph bitmap to be corrected.
                 */
                virtual void CorrectGlyphBitmap(GlyphBitmap& glyph) const = 0;

                /**
                 *  Advance to the next glyph.
                 */
                virtual void Advance() = 0;

                /**
                 *  Verify whether there is any glyph left for iteration.
                 *  @return True if the end of the iteration was reached.
                 */
                virtual bool IsEnd() const = 0;

                static void SetDefaultShaperInstance(Shaper* shaper);

                static Shaper& GetDefaultShaperInstance();

                static Shaper& GetShaperInstance(ShaperType::Enum shaperType);

                /**
                * Cleanup shall clear all the reserved/remaining data.
                * E.g. After initialization content is stored in a shaped way.
                * Normally this is cleaned up/reused with next initialization call.
                * However, it is sometimes useful to free all resources.
                * E.g. remove all references to fonts, delete preallocated arrays.
                * Note: This functionality is by default not thread-safe.
                * This has to be considered when calling or implementing this function.
                * A good approach is to use Candera::TextRendering::Internal::TextRenderLocks::GetGlyphLock()
                * as lock. This lock will be used by a shaper-run, tool.
                */
                virtual void CleanUp() {}

            }; // class Shaper

            /** @} */ // end of CanderaTextEngine

        }// namespace Internal
    }// namespace TextRendering
}// namespace Candera


#endif// Candera_Shaper_h
