//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Parser.h"
#include <Candera/TextEngine/Internal/StyleTools.h>
#include <Candera/TextEngine/Font.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);
    namespace TextRendering {
        namespace Internal {

Parser::Parser(const Style& style, ParserData& parserData) :
    m_isMandatoryLineBreak(false),
    m_isWordBreak(false),
    m_isWordBreakEnabled(false),
    m_produceCorrectChunkOrder(true),
    m_lineBreakType(UndefinedLineBreak),
    m_direction(255),
    m_style(&style),
    m_parserData(&parserData),
    m_lineLength(0),
    m_lineStart(0),
    m_textRemainingLength(0),
    m_chunkLength(0),
    m_chunkStart(0),
    m_unidirectionalChunkLength(0),
    m_unidirectionalChunkStart(0),
    m_lineBreakContext(),
    m_wordBreakContext()
{
}

bool Parser::Initialize(bool isWordBreakEnabled)
{
    m_isMandatoryLineBreak = false;
    m_isWordBreak = false;

    m_lineBreakType = UndefinedLineBreak;

    m_lineStart = TextPointer(0);
    m_lineLength = 0;

    m_textRemainingLength = m_parserData->m_size;
    if (m_textRemainingLength < 0) {
        m_textRemainingLength = 
            static_cast<TextLength>(
                FeatStd::Internal::TextEncoding::CharCount(
                    m_lineStart.GetTChar(m_parserData->m_text)));
    }
    else {
        m_textRemainingLength = static_cast<TextLength>(FeatStd::Internal::TextEncoding::NCharCount(
                                                        m_lineStart.GetTChar(m_parserData->m_text),
                                                        m_textRemainingLength));
    }

    m_chunkStart = TextPointer(0);
    m_chunkLength = 0;
    m_direction = 255;

    m_unidirectionalChunkLength = 0;
    m_isWordBreakEnabled = isWordBreakEnabled;

#ifdef CANDERA_BIDIRECTIONAL_TEXT_ENABLED
    if (!m_bidiParser.Initialize(m_parserData->m_bidirectionalParserData)) {
        return false;
    }
#endif

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
    if (m_parserData->m_isMultilineEnabled) {
        m_lineBreakContext.ch = 0;
        m_lineBreakContext.lbcCur = Unibreak::LBP_Undefined;
        m_lineBreakContext.lbcLast = Unibreak::LBP_Undefined;
        m_lineBreakContext.lbcNew = Unibreak::LBP_Undefined;
        m_lineBreakContext.lbpLang = 0;
        m_lineBreakContext.nextline = 1;
    }
    if (m_isWordBreakEnabled) {
        m_wordBreakContext.wbcSeqStart = Unibreak::WBP_Undefined;
        m_wordBreakContext.wbcLast = Unibreak::WBP_Undefined;
    }
#endif
    return true;
}

FontIdentifier Parser::GetFontIdentifier() const
{
    return StyleTools::GetFontIdentifierByCodePoint(*m_style,
        *FullBoundCodePointIterator(
            m_chunkStart.GetTChar(m_parserData->m_text),
            m_chunkLength));
}

void Parser::AdvanceLineBreak(LineBreakType type)
{
    if (m_textRemainingLength <= 0) {
        FEATSTD_LOG_WARN("Advancing past the end of the text");
        return;
    }

    bool isWrongState = 
        (m_lineBreakType != UndefinedLineBreak) &&
        (m_lineBreakType != type);
    if (isWrongState) {
        FEATSTD_LOG_WARN("Line break type cannot be switched.");
        return;
    }

    m_lineBreakType = type;
    TextPointer lineEnd = m_lineStart + m_lineLength;

    FullBoundCodePointIterator it(lineEnd.GetTChar(m_parserData->m_text), m_textRemainingLength);
    Utf32 character = *it;

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
    Int maximumLineBreakType = 0;
    if (m_parserData->m_isMultilineEnabled) {
        switch(type) {
            case MandatoryLineBreak:
                maximumLineBreakType = LINEBREAK_MUSTBREAK;
                break;
            case AllowLineBreak:
                maximumLineBreakType = LINEBREAK_ALLOWBREAK;
                break;
            case NoLineBreak:
                maximumLineBreakType = LINEBREAK_NOBREAK;
                break;
            case UndefinedLineBreak:
            default:
                maximumLineBreakType = LINEBREAK_INSIDEACHAR;
                break;
        }
        static_cast<void>(is_line_breakable_ex(character, &m_lineBreakContext, 0));
    }
#endif //CANDERA_MULTILINE_TEXT_ENABLED

    bool isBreak = false;
    while (!isBreak) {
        TextLength charSize = static_cast<TextLength>(it.Advance());
        m_lineLength += charSize;
        m_textRemainingLength -= charSize;
        character = *it;

        isBreak = (character == 0);
        m_isMandatoryLineBreak = true;

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
        if ((!isBreak) && m_parserData->m_isMultilineEnabled) {
            Int lineBreakType = is_line_breakable_ex(character, &m_lineBreakContext, 0);
            isBreak = (lineBreakType <= maximumLineBreakType);

            m_isMandatoryLineBreak = (lineBreakType == LINEBREAK_MUSTBREAK);
        }
#endif //CANDERA_MULTILINE_TEXT_ENABLED
    }
}

void Parser::SetLineBreak(const Parser& src)
{
    m_lineStart = src.m_lineStart;
    m_lineLength = src.m_lineLength;
    m_lineBreakType = src.m_lineBreakType;
    m_textRemainingLength = src.m_textRemainingLength;

    m_isMandatoryLineBreak = src.IsMandatoryLineBreak();

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
    m_lineBreakContext = src.m_lineBreakContext;
#endif //CANDERA_MULTILINE_TEXT_ENABLED
}

void Parser::StartLineParsing(bool produceCorrectChunkOrdering)
{
#ifdef CANDERA_BIDIRECTIONAL_TEXT_ENABLED
    m_bidiParser.SetNewLine(m_lineStart, m_lineLength, produceCorrectChunkOrdering);
    m_bidiParser.Advance();
    if (!m_bidiParser.IsEnd()) {
        m_unidirectionalChunkLength = m_bidiParser.GetLength();
    }
    else {
        m_unidirectionalChunkLength = 0;
    }
#else
    m_unidirectionalChunkStart = m_lineStart;
    m_unidirectionalChunkLength = m_lineLength;
#endif

    m_produceCorrectChunkOrder = produceCorrectChunkOrdering;

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
    //reset word break to new line.
    if(m_isWordBreakEnabled) {
        static_cast<void>(is_wordbreak_ex('\n', &m_wordBreakContext, 0));
    }
#endif //CANDERA_MULTILINE_TEXT_ENABLED

    AdvanceToNextChunk();
}

void Parser::AdvanceToNextLine()
{
    m_lineStart += m_lineLength;
    m_lineLength = 0;

    m_lineBreakType = UndefinedLineBreak;
}

void Parser::AdvanceToNextChunk()
{
    if (m_lineBreakType == UndefinedLineBreak) {
        FEATSTD_LOG_WARN("Failed parsing undefined line type.");
        return;
    }

    m_chunkStart = TextPointer(0);
    m_chunkLength = 0;

    if (m_unidirectionalChunkLength <= 0) {
        return;
    }

#ifdef CANDERA_BIDIRECTIONAL_TEXT_ENABLED
    bool resetBidiChunk = 
        (!m_bidiParser.IsEnd()) &&
        (m_unidirectionalChunkLength == m_bidiParser.GetLength());
    // Retrieve the properties of the unidirectional chunk.
    // Length is already stored.
    if (resetBidiChunk) {
        m_unidirectionalChunkStart = m_bidiParser.GetStartPosition();
        m_direction = m_bidiParser.GetDirection();
    }
#endif

    FullBoundCodePointIterator it(m_unidirectionalChunkStart.GetTChar(m_parserData->m_text), m_unidirectionalChunkLength);
    Utf32 character = *it;
    FontIdentifier fontId = StyleTools::GetFontIdentifierByCodePoint(*m_style, character);

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
    // Store unibreak context, in case next advancement start at the same location.
    Unibreak::wordBreakContext wordBreakContext;
    wordBreakContext.wbcSeqStart = Unibreak::WBP_Undefined;
    wordBreakContext.wbcLast = Unibreak::WBP_Undefined;
    if (m_isWordBreakEnabled) {
        wordBreakContext = m_wordBreakContext;
        //return value ignored, just to init wordBreakContext
        static_cast<void>(is_wordbreak_ex(character, &wordBreakContext, 0));
    }
#endif //CANDERA_MULTILINE_TEXT_ENABLED

    TextPointer chunkStart = m_unidirectionalChunkStart;
    TextPointer chunkEnd = m_unidirectionalChunkStart;
    bool foundChunk = false;
    while (!foundChunk) {
        TextLength charSize = static_cast<TextLength>(it.Advance());
        chunkEnd += charSize;
        character = *it;

        bool isBreak = false;
        bool isEndOfUnidirectionalChunk = false;

        m_isWordBreak = false;

        if (character == 0){
            isEndOfUnidirectionalChunk = true;
            // Avoid processing wordbreak when character is 0.
            // Wordbreak is decided by linebreak status.
            if (m_lineBreakType != NoLineBreak) {
                m_isWordBreak = true;
            }
        }
        else {
        // Break at font change.
            FontIdentifier oldFontId = fontId;
            fontId = StyleTools::GetFontIdentifierByCodePoint(*m_style, character);
            isBreak = (fontId != oldFontId);

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
            // Break at word break.
            if ((!isBreak) && m_isWordBreakEnabled) {
                Int wordBreakType = is_wordbreak_ex(character, &wordBreakContext, 0);
                isBreak = (wordBreakType == WORDBREAK_BREAK);
                m_isWordBreak = isBreak;
            }
#endif //CANDERA_MULTILINE_TEXT_ENABLED
        }

        // Break at end of unidirectional chunk.
        isBreak = isBreak || isEndOfUnidirectionalChunk;
        if (isBreak) {
            if ((m_direction == 1) && (!isEndOfUnidirectionalChunk) && m_produceCorrectChunkOrder) {
                // Right to left. Must go to last chunk.
                chunkStart = chunkEnd;
            }
            else {
                // Found the correct chunk. (first for LTR, last for RTL).
                foundChunk = true;
            }
        }
    }

    // Store chunk and update remaining objects.
    m_chunkStart = chunkStart;
    m_chunkLength = chunkEnd - chunkStart;

    m_unidirectionalChunkLength -= m_chunkLength;

    if ((m_direction != 1) || (!m_produceCorrectChunkOrder)) {
        // Left to right. Set next iteration from here.
        m_unidirectionalChunkStart += m_chunkLength;

#ifdef CANDERA_MULTILINE_TEXT_ENABLED
        // Store unibreak context, in case next advancement start at the same location.
        if (m_isWordBreakEnabled) {
            m_wordBreakContext = wordBreakContext;
        }
#endif //CANDERA_MULTILINE_TEXT_ENABLED
    }

#ifdef CANDERA_BIDIRECTIONAL_TEXT_ENABLED
    // Store the size of the next unidirectional chunk,
    // so that parsing of current line does not end.
    if (m_unidirectionalChunkLength <= 0) {
        m_bidiParser.Advance();
        if (!m_bidiParser.IsEnd()) {
            m_unidirectionalChunkLength = m_bidiParser.GetLength();
        }
    }
#endif
}

        }
    }
}
