//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "HbUnicode.h"
#include <Candera/TextEngine/Internal/LookupTools.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

namespace Candera { namespace TextRendering { namespace Internal { namespace HarfBuzzShaper {

hb_unicode_combining_class_t HbUnicodeCombiningClass(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t unicode,
    void* /*user_data*/)
{
    return HbUnicode::FindCombiningClass(unicode);
}

HbInterfaceUnsignedInt HbUnicodeEastasianWidth(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t unicode,
    void* /*user_data*/)
{
    return (HbUnicode::IsEastAsianWidthWide(unicode) != 0) ? 2 : 1;
}

hb_unicode_general_category_t HbUnicodeGeneralCategory(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t unicode,
    void* /*user_data*/)
{
    return HbUnicode::FindGeneralCategory(unicode);
}

hb_codepoint_t HbUnicodeMirroring(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t unicode,
    void* /*user_data*/)
{
    return HbUnicode::FindBidiMirror(unicode);
}

hb_script_t HbUnicodeScript(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t unicode,
    void* /*user_data*/)
{
    return HbUnicode::FindScript(unicode);
}

hb_bool_t HbUnicodeCompose(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t a,
    hb_codepoint_t b,
    hb_codepoint_t *ab,
    void* /*user_data*/)
{
    return HbUnicode::Compose(ab, a, b);
}

hb_bool_t HbUnicodeDecompose(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t ab,
    hb_codepoint_t *a,
    hb_codepoint_t *b,
    void* /*user_data*/)
{
    return HbUnicode::Decompose(ab, a, b);
}

HbInterfaceUnsignedInt HbUnicodeDecomposeCompatibility(
    hb_unicode_funcs_t* /*ufuncs*/,
    hb_codepoint_t u,
    hb_codepoint_t *decomposed,
    void* /*user_data*/)
{
    decomposed[0] = u;
    return 1;
}
    
void HbUnicode::SetFuncs(hb_buffer_t* buffer)
{
    hb_unicode_funcs_t *funcs = CreateFuncs();
    hb_buffer_set_unicode_funcs(buffer, funcs);
    hb_unicode_funcs_destroy(funcs);
}

hb_unicode_combining_class_t HbUnicode::FindCombiningClass(hb_codepoint_t codepoint)
{
    const CombiningClassData* start = g_combiningClassTable;
    const CombiningClassData* end = g_combiningClassTable + g_combiningClassTableSize;
    const CombiningClassData* result = LookupTools::FindRange(start, end, codepoint);
    if (result == end) {
        return HB_UNICODE_COMBINING_CLASS_NOT_REORDERED;
    }
    else {
        return static_cast<hb_unicode_combining_class_t>(result->combiningClass);
    }
}

hb_bool_t HbUnicode::IsEastAsianWidthWide(hb_codepoint_t codepoint)
{
    const EastAsianWidthData* start = g_eastAsianWidthTable;
    const EastAsianWidthData* end = g_eastAsianWidthTable + g_eastAsianWidthTableSize;
    const EastAsianWidthData* result = LookupTools::FindRange(start, end, codepoint);
    return (result != end);
}

hb_unicode_general_category_t HbUnicode::FindGeneralCategory(hb_codepoint_t codepoint)
{
    const GeneralCategoryData* start = g_generalCategoryTable;
    const GeneralCategoryData* end = g_generalCategoryTable + g_generalCategoryTableSize;
    const GeneralCategoryData* result = LookupTools::FindRange(start, end, codepoint);
    if (result == end) {
        return HB_UNICODE_GENERAL_CATEGORY_CONTROL;
    }
    else {
        return result->generalCategory;
    }
}

hb_codepoint_t HbUnicode::FindBidiMirror(hb_codepoint_t codepoint)
{
    const BidiMirroringData* start = g_bidiMirroringTable;
    const BidiMirroringData* end = g_bidiMirroringTable + g_bidiMirroringTableSize;
    const BidiMirroringData* result = LookupTools::Find1(start, end, codepoint);
    if (result == end) {
        return codepoint;
    }
    else {
        return result->mirror;
    }
}

hb_script_t HbUnicode::FindScript(hb_codepoint_t codepoint)
{
    const ScriptData* start = g_scriptTable;
    const ScriptData* end = g_scriptTable + g_scriptTableSize;
    const ScriptData* result = LookupTools::FindRange(start, end, codepoint);
    if (result == end) {
        return HB_SCRIPT_UNKNOWN;
    }
    else {
        return result->script;
    }
}

hb_bool_t HbUnicode::Compose(hb_codepoint_t* ab, hb_codepoint_t a, hb_codepoint_t b)
{
    CanonicalCompositionData point = {a, b, 0};
    const CanonicalCompositionData* start = g_canonicalCompositionTable;
    const CanonicalCompositionData* end = g_canonicalCompositionTable + g_canonicalCompositionTableSize;
    const CanonicalCompositionData* result = LookupTools::Find2(start, end, &point);
    if (result == end) {
        return false;
    }
    else {
        *ab = result->value;
        return true;
    }
}

hb_bool_t HbUnicode::Decompose(hb_codepoint_t ab, hb_codepoint_t *a, hb_codepoint_t *b)
{
    const CanonicalDecompositionData* start = g_canonicalDecompositionTable;
    const CanonicalDecompositionData* end = g_canonicalDecompositionTable + g_canonicalDecompositionTableSize;
    const CanonicalDecompositionData* result = LookupTools::Find1(start, end, ab);
    if (result == end) {
        return false;
    }
    else {
        *a = result->first;
        *b = result->second;
        return true;
    }
}

hb_unicode_funcs_t *HbUnicode::CreateFuncs ()
{
    hb_unicode_funcs_t *funcs = hb_unicode_funcs_create(0);
    if (funcs != 0) {
        hb_unicode_funcs_set_combining_class_func(funcs, HbUnicodeCombiningClass, 0, 0);
        hb_unicode_funcs_set_eastasian_width_func(funcs, HbUnicodeEastasianWidth, 0, 0);
        hb_unicode_funcs_set_general_category_func(funcs, HbUnicodeGeneralCategory, 0, 0);
        hb_unicode_funcs_set_mirroring_func(funcs, HbUnicodeMirroring, 0, 0);
        hb_unicode_funcs_set_script_func(funcs, HbUnicodeScript, 0, 0);
        hb_unicode_funcs_set_compose_func(funcs, HbUnicodeCompose, 0, 0);
        hb_unicode_funcs_set_decompose_func(funcs, HbUnicodeDecompose, 0, 0);
        hb_unicode_funcs_set_decompose_compatibility_func(funcs, HbUnicodeDecomposeCompatibility, 0, 0);
    }

    return funcs;
}

}}}} // current namespace

