//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "HbShaper.h"
#include <Candera/TextEngine/Internal/HarfBuzzShaper/HbUnicode.h>

#include <FeatStd/Diagnostics/Debug.h>
#include <Candera/TextEngine/Internal/GlyphRenderer.h>
#include <Candera/TextEngine/Internal/ShaperInfo.h>
#include <Candera/TextEngine/Font.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <Candera/TextEngine/TextRenderer.h>
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera { namespace TextRendering { namespace Internal { namespace HarfBuzzShaper {

static hb_direction_t HbDirectionFromShaperInfo(GlyphBitmap::Direction direction)
{
    hb_direction_t ret;
    switch(direction) {
        case GlyphBitmap::LeftToRight: ret = HB_DIRECTION_LTR;
            break;
        case GlyphBitmap::RightToLeft: ret = HB_DIRECTION_RTL;
            break;
        default: ret = HB_DIRECTION_INVALID;
            break;
    }
    return ret;
}

HbShaper::HbShaper():
    m_numGlyph(0),
    m_currentGlyph(0),
    m_glyph(0),
    m_position(0),
    m_buffer(0)
{
    m_buffer = hb_buffer_create();
    HbUnicode::SetFuncs(m_buffer);
}

HbShaper::~HbShaper()
{
    if (m_buffer != 0) {
        hb_buffer_destroy(m_buffer);
    }
    m_buffer = 0;
    m_glyph = 0;
    m_position = 0;
}

bool HbShaper::Initialize(const GlyphRenderer& glyphRenderer, const Font &font, const TChar* text, TextLength textSize, const ShaperInfo& info)
{
    if (m_buffer == 0) {
        m_buffer = hb_buffer_create();
        HbUnicode::SetFuncs(m_buffer);
    }
    hb_buffer_clear_contents(m_buffer);
    m_numGlyph = 0;
    m_currentGlyph = 0;

    if (text == 0) {
        return false;
    }

    m_font.Initialize(glyphRenderer, font);
    if (m_font.GetHbFont() == 0) {
        return false;
    }

    hb_buffer_set_direction(m_buffer, HbDirectionFromShaperInfo(info.GetDirection()));
    /* setup buffer */
    if (info.GetLocale() != 0) {
        // TODO: Review locale strings.
        hb_buffer_set_language(m_buffer, hb_language_from_string (info.GetLocale(), -1));
    }
    FEATSTD_COMPILETIME_ASSERT((sizeof(TChar) == sizeof(HbInterfaceChar)) && (TCharIsUtfEncoded != 0));
    hb_buffer_add_utf8(m_buffer, text, textSize, 0, textSize);

    // Guessing of missing properties is no longer automatic.
    hb_buffer_guess_segment_properties(m_buffer);

    const Char* featureStrings[] = {
#ifdef CANDERA_TEXTENGINE_HARFBUZZ_CLIG_FEATURE_ENABLED
        "clig = 1",
#else
        "clig = 0",
#endif
#ifdef CANDERA_TEXTENGINE_HARFBUZZ_DLIG_FEATURE_ENABLED
        "dlig = 1",
#else
        "dlig = 0",
#endif
#ifdef CANDERA_TEXTENGINE_HARFBUZZ_LIGA_FEATURE_ENABLED
        "liga = 1",
#else
        "liga = 0",
#endif
#ifdef CANDERA_TEXTENGINE_HARFBUZZ_LOCL_FEATURE_ENABLED
        "locl = 1"
#else
        "locl = 0"
#endif
    };
    const UInt featureArraySize = sizeof(featureStrings) / sizeof(featureStrings[0]);
    hb_feature_t features[featureArraySize];
    UInt featureCount = 0;
    for (UInt i = 0; i < featureArraySize; i++) {
        if (hb_feature_from_string(featureStrings[i], -1, &features[featureCount]) != 0) {
            featureCount++;
        }
    }

    hb_shape(m_font.GetHbFont(), m_buffer, features, featureCount);

    /* buffer output */
    m_numGlyph = hb_buffer_get_length (m_buffer);
    m_glyph = hb_buffer_get_glyph_infos (m_buffer, 0);
    m_position = hb_buffer_get_glyph_positions (m_buffer, 0);

    if ((info.GetDirection() == GlyphBitmap::RightToLeft) && (info.GetGlyphOrder() == TextRenderContext::OriginalGlyphOrder)) {
        for (Int i = 0; (i * 2) < m_numGlyph; i++) {
            hb_glyph_info_t glyph = m_glyph[i];
            hb_glyph_position_t position = m_position[i];
            m_glyph[i] = m_glyph[(m_numGlyph - i) - 1];
            m_position[i] = m_position[(m_numGlyph - i) - 1];
            m_glyph[(m_numGlyph - i) - 1] = glyph;
            m_position[(m_numGlyph - i) - 1] = position;
        }
    }

    return true;
}

GlyphIndex HbShaper::GetGlyph() const
{
    if (!IsEnd()) {
        return static_cast<GlyphIndex>(m_glyph[m_currentGlyph].codepoint);
    }
    return 0;
}

GlyphBitmap::Direction HbShaper::GetDirection() const
{
    return (hb_buffer_get_direction(m_buffer) == HB_DIRECTION_LTR) ? GlyphBitmap::LeftToRight : GlyphBitmap::RightToLeft;
}

TextPosition HbShaper::GetCharacterPosition() const
{
    if (!IsEnd()) {
        return static_cast<TextPosition>(m_glyph[m_currentGlyph].cluster);
    }
    return 0;
}

void HbShaper::CorrectGlyphBitmap(GlyphBitmap& glyph) const
{
    if (!IsEnd()) {
        glyph.xadvance = static_cast<GlyphBitmap::AdvanceType>(m_position[m_currentGlyph].x_advance);
        glyph.yadvance = static_cast<GlyphBitmap::AdvanceType>(m_position[m_currentGlyph].y_advance);

        glyph.left += static_cast<GlyphBitmap::AdvanceType>(m_position[m_currentGlyph].x_offset);
        glyph.top -= static_cast<GlyphBitmap::AdvanceType>(m_position[m_currentGlyph].y_offset);
    }
}

void HbShaper::Advance()
{
    m_currentGlyph++;
}

bool HbShaper::IsEnd() const
{
    return m_currentGlyph >= m_numGlyph;
}


void HbShaper::CleanUp()
{
#ifdef FEATSTD_THREADSAFETY_ENABLED
    FeatStd::Internal::CriticalSectionLocker lock(Candera::TextRendering::Internal::TextRenderLocks::GetGlyphLock());
#endif
    m_font.Finalize();
    if (m_buffer != 0) {
        hb_buffer_destroy(m_buffer);
    }
    m_buffer = 0;
    m_glyph = 0;
    m_position = 0;
    m_numGlyph = 0;
    m_currentGlyph = 0;
}

}}}} // current namespace
