//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_BidirectionalParserData_h)
#define Candera_BidirectionalParserData_h

#include <Candera/Environment.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/Internal/TextPointer.h>

namespace Candera {
    namespace TextRendering {
        namespace Internal {
/** @addtogroup CanderaTextEngine
 *  @{
 */
/**
 *  The text BidirectionalParserData splits a text into homogeneous
 *  chunks of text that have a unique processing direction.
 *  It can also sort lines of text to evaluate in proper order.
 */
class BidirectionalParserData {
public:
    /**
     * Constructor.
     */
    BidirectionalParserData();

    /**
     * Destructor.
     */
    ~BidirectionalParserData();

    /**
     * Initialize the parser.
     *
     * Initialize the parser with the given text.
     * @param text Text to be parsed.
     * @param size Size of text to be parsed.
     * @param baseLevel The base embeding level for the paragraphs.
     * @param overrideParagraphBreak When enabled, ignore paragraph breaks within the text.
     */
    bool Initialize(const TChar* text, TextLength size, UInt8 baseLevel, bool overrideParagraphBreak);

    /**
     * Update a text line.
     *
     * This is called line by line. Text outside the bounds processed
     * by this functions should not accessed. Levels might be invalid.
     * @param start Start address of the processed text.
     * @param size Size of text to be parsed.
     * @param generateLineLevels When false, prevents the update of 
     *                           the embedding levels of spaces near 
     *                           the end of the line.
     */
    bool UpdateLine(TextPointer start, TextLength size, bool generateLineLevels);

    /**
     * Get the start position of the parsed text.
     */
    const TChar* GetStartPosition() const { return m_start; }

    /**
     * Get the end position of the parsed text.
     */
    const TChar* GetEndPosition() const { return m_start + m_measure.m_TCharSize; }

    /**
     * Get the length of the parsed text, in TChars.
     */
    TextLength GetLength() const { return m_measure.m_TCharSize; }

    /**
     * Get the length of the parsed text.
     */
    TextLength GetCodePointCount() const { return m_measure.m_CodePointSize; }

    /**
     * Get a buffer storing bidirectional class data of the parsed text.
     */
    UInt8* GetClassBuffer();

    /**
     * Get a buffer storing embedding level data of the parsed text.
     */
    UInt8* GetLevelBuffer();

private:
    struct Measure{
        TextLength m_TCharSize;
        TextLength m_CodePointSize;
    };
    Measure MeasureText(const TChar* start, TextLength maxSize) const;
    TextPointer FindEndOfParagraph(TextPointer start, TextPointer end) const;

    UInt8 m_baseLevel;
    TextPointer m_lastLineStart;
    TextLength m_lastLineLenght;

    const TChar* m_start;
    UInt8* m_bigBuffer;

    Measure m_measure;
};

 /** @} */ // end of CanderaTextEngine

        }     // namespace Internal
    }    // namespace TextRendering
}    // namespace Candera

#endif    // Candera_BidirectionalParserData_h
