//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BidirectionalParserData.h"
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <Candera/TextEngine/Internal/BidirectionalParser/BidiUnicode.h>
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>

namespace Candera { namespace TextRendering { namespace Internal {

BidirectionalParserData::BidirectionalParserData() :
    m_baseLevel(255),
    m_lastLineStart(0),
    m_lastLineLenght(0),
    m_start(0),
    m_bigBuffer(0),
    m_measure()
{
}

BidirectionalParserData::~BidirectionalParserData()
{
    CANDERA_DELETE_ARRAY(m_bigBuffer);
    m_bigBuffer = 0;
    m_start = 0;
}

bool BidirectionalParserData::Initialize(const TChar* text, TextLength size, UInt8 baseLevel, bool overrideParagraphBreak)
{
    m_measure = MeasureText(text, size);

    m_start = text;
    TextPointer end = TextPointer(static_cast<TextPosition>(m_measure.m_TCharSize));

    //text not empty
    if (GetLength() > 0) {

        //TO DO: Use stack allocated buffer for small strings.
        CANDERA_DELETE_ARRAY(m_bigBuffer);
        m_bigBuffer = CANDERA_NEW_ARRAY(UInt8, static_cast<SizeType>(GetLength()) * 2U);
        if (m_bigBuffer == 0) {
            return false;
        }

        TextPointer paragraphEnd = TextPointer(0);
        TextPosition paragraphCharStart = 0;
        while (paragraphEnd < end) {
            TextPointer paragraphStart = paragraphEnd;
            if (overrideParagraphBreak) {
                paragraphEnd = end;
            }
            else {
                paragraphEnd = FindEndOfParagraph(paragraphStart, end);
            }
            paragraphCharStart += BidiUnicode::GenerateParagraphLevels(
                GetLevelBuffer() + paragraphCharStart,
                GetClassBuffer() + paragraphCharStart,
                m_start,
                paragraphStart,
                paragraphEnd,
                baseLevel
            );
        }

    }

    m_baseLevel = baseLevel;
    m_lastLineStart = TextPointer(0);
    m_lastLineLenght = 0;

    return true;
}

bool BidirectionalParserData::UpdateLine(TextPointer start, TextLength size, bool generateLineLevels)
{
    if (m_start == 0) {
        return false;
    }

    if (start.GetPosition() > m_measure.m_TCharSize) {
        return false;
    }

    TextLength remainingTSize = m_measure.m_TCharSize - start.GetPosition();
    if ((size < 0) || (size > remainingTSize)) {
        size = remainingTSize;
    }

    // Parsers may be copied, so the line might be updated more then once for
    // the same settings. Avoid this.
    if ((m_lastLineStart == start) && (m_lastLineLenght == size)) {
        return true;
    }

    if (generateLineLevels) {
        m_lastLineStart = start;
        m_lastLineLenght = size;

        TextPosition codePointPosition = 
            TextPointer::GetCodePointPosition(m_start, TextPointer(0), start);

        TextLength processed = BidiUnicode::GenerateLineLevels(
            GetLevelBuffer() + codePointPosition,
            GetClassBuffer() + codePointPosition,
            m_start,
            start,
            start + size,
            m_baseLevel
        );
        FEATSTD_DEBUG_ASSERT((processed + codePointPosition) <= m_measure.m_CodePointSize);
        FEATSTD_RELEASE_UNUSED(processed);
    }
    return true;
}

TextPointer BidirectionalParserData::FindEndOfParagraph(TextPointer start, TextPointer end) const
{
    FullBoundCodePointIterator it(start.GetTChar(m_start), end - start);
    CodePoint codepoint;
    do {
        codepoint = *it;
        start += static_cast<TextLength>(it.Advance());
    }
    while ((codepoint != 0) && (BidiUnicode::GetBidiClass(codepoint) != BidiUnicode::B));
    return start;
}

BidirectionalParserData::Measure BidirectionalParserData::MeasureText(const TChar* start, TextLength maxSize) const
{
    Measure measure = { 0, 0};
        FullBoundCodePointIterator it(start, maxSize);
        CodePoint codepoint = (*it);
        do {
            if (codepoint != 0) {
                 measure.m_CodePointSize ++;
            }
            measure.m_TCharSize += static_cast<TextLength>(it.Advance());
            codepoint = (*it);
        } while (codepoint != 0);
    return measure;
}
        
UInt8* BidirectionalParserData::GetClassBuffer()
{
    return m_bigBuffer;
}

UInt8* BidirectionalParserData::GetLevelBuffer()
{
    return m_bigBuffer + GetLength();
}

}}} //namespaces.
