//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_BidirectionalParser_h)
#define Candera_BidirectionalParser_h

#include <Candera/Environment.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/Internal/BidirectionalParser/BidiUnicode.h>
#include <Candera/TextEngine/Internal/BidirectionalParser/BidirectionalParserData.h>
#include <Candera/TextEngine/Internal/TextPointer.h>

namespace Candera {
    namespace TextRendering {
        namespace Internal {
/** @addtogroup CanderaTextEngine
 *  @{
 */
            /**
             *  The text BidirectionalParser splits a text into homogeneous
             *  chunks of text that have a unique processing direction.
             *  It can also sort lines of text to evaluate in proper order.
             */
            class BidirectionalParser {
            public:
                /**
                 * Constructor.
                 */
                BidirectionalParser();

                /**
                 * Initialize the parser.
                 *
                 * Initialize the parser with the given data.
                 * @param data Data object that was initialized with the text being parsed.
                 */
                bool Initialize(BidirectionalParserData& data);

                /**
                 * Initialize the parser with a piece of text to be iterated.
                 * The text needs to be a subset of the text provided to the
                 */
                void SetNewLine(TextPointer startTChar, TextLength lengthTChar, bool bidirectionalProduction);

                /**
                 * Get the start position of the current chunk.
                 */
                TextPointer GetStartPosition() const { return m_chunkTCharStart; }

                /**
                 * Get current the length of the current chunk.
                 */
                TextLength GetLength() const { return m_chunkTCharEnd - m_chunkTCharStart; }

                /**
                 * Get the text direction of the current chunk.
                 * @return 0 left to right, 1 right to left
                 */
                UInt8 GetDirection() const { return m_chunkLevel & 1U; }

                /**
                 * Advance to the next chunk in left to right order.
                 */
                void Advance();

                /**
                 * Determine whether parsing is complete.
                 */
                bool IsEnd() const { return (m_chunkTCharEnd == m_chunkTCharStart); }

            private:
                TextPointer GetTCharPosition(TextPointer start, TextPosition position) const;
                TextPosition GetCodePointPosition(TextPointer start, TextPointer position) const;

                struct Chunk {
                    UInt8 m_level;
                    TextPosition m_start;
                    TextPosition m_end;
                };

                static Chunk GetChunkFromStart(TextPosition start, TextPosition end, const UInt8* levels);
                static Chunk GetChunkFromEnd(TextPosition start, TextPosition end, const UInt8* levels);

                static Chunk GetChunkFromRightEmbedding(TextPosition start, TextPosition end, UInt8 level, const UInt8* levels);
                static Chunk GetChunkFromLeftEmbedding(TextPosition start, TextPosition end, UInt8 level, const UInt8* levels);
                static Chunk GetChunkFromEmbedding(TextPosition start, TextPosition end, UInt8 level, const UInt8* levels);

                static Chunk GetChunkOutsideEmbedding(TextPosition start, TextPosition end, UInt8 level, TextPosition marginalStart, TextPosition marginalEnd, const UInt8* levels);

                bool m_isBidirectionalProduction;
                
                UInt8 m_chunkLevel;

                BidirectionalParserData* m_data;

                TextPointer m_chunkTCharStart;
                TextPointer m_chunkTCharEnd;

                TextPosition m_chunkStart;
                TextPosition m_chunkEnd;

                TextPointer m_lineTCharStart;
                TextLength m_lineTCharLength;
                TextPosition m_lineStart;
                TextPosition m_lineEnd;
            };

 /** @} */ // end of CanderaTextEngine

        }     // namespace Internal
    }    // namespace TextRendering
}    // namespace Candera

#endif    // Candera_BidirectionalParser_h
