//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BidirectionalParser.h"
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>

#include <FeatStd/Diagnostics/Debug.h>

namespace Candera { namespace TextRendering { namespace Internal {
    BidirectionalParser::BidirectionalParser():
        m_isBidirectionalProduction(false),
        m_chunkLevel(0),
        m_data(0),
        m_chunkTCharStart(0),
        m_chunkTCharEnd(0),
        m_chunkStart(0),
        m_chunkEnd(0),
        m_lineTCharStart(0),
        m_lineTCharLength(0),
        m_lineStart(0),
        m_lineEnd(0)
    {
    }

    bool BidirectionalParser::Initialize(BidirectionalParserData& data)
    {
        m_isBidirectionalProduction = false;
        m_chunkLevel = 0;
        m_chunkTCharStart = TextPointer(0);
        m_chunkTCharEnd = TextPointer(0);
        m_chunkStart = 0;
        m_chunkEnd = 0;
        m_lineTCharStart = TextPointer(0);
        m_lineTCharLength = 0;
        m_lineStart = 0;
        m_lineEnd = 0;

        m_data = &data;
        return true;
    }

    void BidirectionalParser::SetNewLine(TextPointer startTChar, TextLength lengthTChar, bool bidirectionalProduction)
    {
        if (m_data == 0) {
            return;
        }

        if (!m_data->UpdateLine(startTChar, lengthTChar, bidirectionalProduction)){
            return;
        }

        m_chunkLevel = 0;

        m_isBidirectionalProduction = bidirectionalProduction;

        FEATSTD_DEBUG_ASSERT(TextPointer(0) <= startTChar);
        TextPointer endTChar = startTChar + lengthTChar;
        FEATSTD_DEBUG_ASSERT(endTChar <= TextPointer(static_cast<TextPosition>(m_data->GetLength())));

        TextPointer startTCharReference = TextPointer(0);
        TextPosition startReference = 0;
        if ((m_lineTCharStart != TextPointer(0)) && ((m_lineTCharStart + m_lineTCharLength) <= startTChar)) {
            startTCharReference = m_lineTCharStart + m_lineTCharLength;
            startReference = m_lineEnd;
        }

        m_lineTCharStart = startTChar;
        m_lineTCharLength = lengthTChar;
        m_lineStart = startReference + GetCodePointPosition(startTCharReference, startTChar);
        m_lineEnd = m_lineStart + GetCodePointPosition(m_lineTCharStart, endTChar);

        FEATSTD_DEBUG_ASSERT((m_lineStart >= 0) && (m_lineEnd <= m_data->GetCodePointCount()));

        m_chunkTCharStart = m_lineTCharStart;
        m_chunkTCharEnd = m_lineTCharStart;
        m_chunkStart = m_lineStart;
        m_chunkEnd = m_lineStart;
    }

    void BidirectionalParser::Advance()
    {
        if (m_data == 0) {
            return;
        }
        if ((m_chunkStart == m_chunkEnd) && (m_chunkStart == m_lineEnd)) {
            return;
        }

        if (!m_data->UpdateLine(m_lineTCharStart, m_lineTCharLength, m_isBidirectionalProduction)){
            return;
        }

        const UInt8* levels = m_data->GetLevelBuffer();

        Chunk chunk = {m_chunkLevel, m_chunkStart, m_chunkEnd};

        if (m_isBidirectionalProduction) {
            // Browse for next chunk.
            Chunk nextChunk;
            // Check if we should climb to higher level embeddings.
            if ((chunk.m_level & 1U) == 0U) {
                // Left to right chunk. Determine the one to the right.
                nextChunk = GetChunkFromRightEmbedding(chunk.m_end, m_lineEnd, chunk.m_level, levels);
            }
            else {
                // Right to left chunk. Determine the one to the left.
                nextChunk = GetChunkFromLeftEmbedding(m_lineStart, chunk.m_start, chunk.m_level, levels);
            }
            if (nextChunk.m_start == nextChunk.m_end) {
                // No higher level found. Need to descend to lower level.
                nextChunk = GetChunkOutsideEmbedding(chunk.m_start, chunk.m_end, chunk.m_level, m_lineStart, m_lineEnd, levels);
            }

            chunk = nextChunk;
        }
        else {
            // Get next chunk in Buffer.
            chunk = GetChunkFromStart(chunk.m_end, m_lineEnd, levels);
        }

        TextPosition previousChunkEnd = m_chunkEnd;

        m_chunkStart = chunk.m_start;
        m_chunkEnd = chunk.m_end;
        m_chunkLevel = chunk.m_level;

        // Retrieve TChar position starting from last chunk if possible.
        TextPointer startTChar = TextPointer(0);
        if (chunk.m_start >= previousChunkEnd) {
            startTChar = m_chunkTCharEnd;
            chunk.m_start -= previousChunkEnd;
            chunk.m_end -= previousChunkEnd;
        }
        m_chunkTCharStart = GetTCharPosition(startTChar, chunk.m_start);
        m_chunkTCharEnd = GetTCharPosition(m_chunkTCharStart, chunk.m_end - chunk.m_start);
    }

    TextPointer BidirectionalParser::GetTCharPosition(TextPointer start, TextPosition position) const
    {
        return TextPointer::GetTCharPosition(m_data->GetStartPosition(), m_data->GetLength(), start, position);
    }

    TextPosition BidirectionalParser::GetCodePointPosition(TextPointer start, TextPointer position) const
    {
        return TextPointer::GetCodePointPosition(m_data->GetStartPosition(), start, position);
    }

    BidirectionalParser::Chunk BidirectionalParser::GetChunkFromStart(TextPosition start, TextPosition end, const UInt8* levels)
    {
        Chunk chunk = { 0, start, start };
        if (chunk.m_end < end) {
            chunk.m_level = levels[chunk.m_end++];
            while((chunk.m_end < end) && (chunk.m_level == levels[chunk.m_end])) {
                ++chunk.m_end;
            }
        }
        return chunk;
    }

    BidirectionalParser::Chunk BidirectionalParser::GetChunkFromEnd(TextPosition start, TextPosition end, const UInt8* levels)
    {
        Chunk chunk = { 0, end, end };
        if (chunk.m_start > start) {
            chunk.m_level = levels[--chunk.m_start];
            while((chunk.m_start > start) && (chunk.m_level == levels[--chunk.m_start])) {}
            if (chunk.m_level != levels[chunk.m_start]) {
                ++chunk.m_start;
            }
        }
        return chunk;
    }

    BidirectionalParser::Chunk BidirectionalParser::GetChunkFromRightEmbedding(TextPosition start, TextPosition end, UInt8 level, const UInt8* levels)
    {
        UInt8 rightLevel;
        // Current level is not empty.
        if (start < end) {
            rightLevel = levels[start];
            // First chunk at the right should be at a higher level to have an embedding.
            if (rightLevel > level) {
                // Determine the level at right.
                // The embedding level is the lowest level higher than the current level.
                TextPosition embeddingEnd = start + 1;
                while((embeddingEnd < end) && (level < levels[embeddingEnd])) {
                    if (rightLevel > levels[embeddingEnd]) {
                        rightLevel = levels[embeddingEnd];
                    }
                    ++embeddingEnd;
                }
                return GetChunkFromEmbedding(start, embeddingEnd, rightLevel, levels);
            }
        }

        // There is no embedding at the right.
        Chunk chunk = { 0, start, start };
        return chunk;
    }

    BidirectionalParser::Chunk BidirectionalParser::GetChunkFromLeftEmbedding(TextPosition start, TextPosition end, UInt8 level, const UInt8* levels)
    {
        UInt8 leftLevel;
        // Current level is not empty.
        if (start < end) {
            leftLevel = levels[end - 1];
            // First chunk at the left should be at a higher level to have an embedding.
            if (leftLevel > level) {
                // Determine the level at left.
                // The embedding level is the lowest level higher than the current level.
                TextPosition embeddingStart = end - 1;
                while((embeddingStart >= start) && (level < levels[embeddingStart])) {
                    if (leftLevel > levels[embeddingStart]) {
                        leftLevel = levels[embeddingStart];
                    }
                    --embeddingStart;
                }
                ++embeddingStart;
                return GetChunkFromEmbedding(embeddingStart, end, leftLevel, levels);
            }
        }

        // There is no embedding at the left.
        Chunk chunk = { 0, end, end };
        return chunk;
    }

    BidirectionalParser::Chunk BidirectionalParser::GetChunkFromEmbedding(TextPosition start, TextPosition end, UInt8 level, const UInt8* levels)
    {
        // Is embedding left to right or right to left.
        if ((level & 1U) == 0U) {
            // Left to right embedding.
            Chunk chunk = GetChunkFromStart(start, end, levels);
            if (chunk.m_level == level) {
                // If the chunk at the start of the embedding is the correct level.
                return chunk;
            }
            FEATSTD_DEBUG_ASSERT(chunk.m_level > level);
            // Go deeper.
            return GetChunkFromRightEmbedding(start, end, level, levels);
        }
        else {
            // Right to left embedding.
            Chunk chunk = GetChunkFromEnd(start, end, levels);
            if (chunk.m_level == level) {
                // If the chunk at the start of the embedding is the correct level.
                return chunk;
            }
            FEATSTD_DEBUG_ASSERT(chunk.m_level > level);
            // Go deeper.
            return GetChunkFromLeftEmbedding(start, end, level, levels);
        }
    }

    BidirectionalParser::Chunk BidirectionalParser::GetChunkOutsideEmbedding(TextPosition start, TextPosition end, UInt8 level, TextPosition marginalStart, TextPosition marginalEnd, const UInt8* levels)
    {
        //Find closest plausible level to the right.
        UInt8 rightLevel = 0;
        TextPosition rightPos = end;
        while(rightPos < marginalEnd) {
            if (level >= levels[rightPos]) {
                rightLevel = levels[rightPos];
                break;
            }
            rightPos++;
        }

        //Find closest plausible level to the left.
        UInt8 leftLevel = 0;
        TextPosition leftPos = start;
        if (leftPos > marginalStart) {
            --leftPos;
            while(leftPos >= marginalStart) {
                if (level >= levels[leftPos]) {
                    leftLevel = levels[leftPos];
                    break;
                }
                --leftPos;
            }
            ++leftPos;
        }

        if (leftLevel == rightLevel) {
            // There are two chunks. Choose the one in the correct direction.
            if ((leftLevel & 1U) == 0U) {
                return GetChunkFromStart(rightPos, marginalEnd, levels);
            }
            else{
                return GetChunkFromEnd(marginalStart, leftPos, levels);
            }
        }
        else if (rightLevel > leftLevel) {
            // Right chunk is the winner.
            if ((rightLevel & 1U) == 0U) {
                // Direction is correct.
                return GetChunkFromStart(rightPos, marginalEnd, levels);
            }
            else {
                //Direction is incorrect, so we have to search farther.
                return GetChunkOutsideEmbedding(leftPos, rightPos, rightLevel - 1, marginalStart, marginalEnd, levels);
            }
        }
        else { // rightLevel < leftLevel
            // Left chunk is the winner.
            if ((leftLevel & 1U) != 0U) {
                // Direction is correct.
                return GetChunkFromEnd(marginalStart, leftPos, levels);
            }
            else {
                //Direction is incorrect, so we have to search farther.
                return GetChunkOutsideEmbedding(leftPos, rightPos, leftLevel - 1, marginalStart, marginalEnd, levels);
            }
        }
    }
}}} //namespaces.
