//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/Environment.h>

#include <Candera/TextEngine/Freetype/FtInclude.h>
#include <Candera/TextEngine/Freetype/FtGlyphRenderer.h>
#include <Candera/TextEngine/Freetype/FtFont.h>

#include <Candera/TextEngine/FontEngine.h>
#include <Candera/TextEngine/TextCoordinate.h>
#include <Candera/TextEngine/TextRenderer.h>

#include <Candera/TextEngine/Internal/GlyphRenderInfo.h>

#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/Internal/Utf32IgnoreList.h>

//#define FT_USE_DEFAULT_GLYPH_CACHE 1

#ifndef FT_USE_DEFAULT_GLYPH_CACHE
#include <Candera/TextEngine/GlyphCacheAccess.h>
#endif

namespace Candera {
    namespace TextRendering {
        namespace Internal {
            // ----------------------------------------------------------------------------
            FtGlyphRenderer::FtGlyphRenderer(const Internal::GlyphRenderInfo& renderInfo) :
                m_renderInfo(renderInfo)
            {
            }
            // ----------------------------------------------------------------------------
            FtGlyphRenderer::~FtGlyphRenderer()
            {
            }
            
            // ----------------------------------------------------------------------------
            GlyphIndex FtGlyphRenderer::GetGlyphIndex(const FtFont &font, Utf32 unicode, bool ignoreDefault) const
            {
                GlyphIndex index = FTC_CMapCache_Lookup(
                    FtFontEngine::CMapCache(), 
                    font.m_type.face_id, 
                    font.m_properties.m_unicodeCharMap, 
                    static_cast<FT_UInt32>(unicode));

                if ((index == 0) && (!ignoreDefault) && (m_renderInfo.defaultGlyph != 0) && (!Utf32IgnoreList::IsIgnoreCode(unicode))) {
                    index = FTC_CMapCache_Lookup(
                    FtFontEngine::CMapCache(),
                    font.m_type.face_id,
                    font.m_properties.m_unicodeCharMap,
                    static_cast<FT_UInt32>(m_renderInfo.defaultGlyph));
                }
                return index;
            }

            GlyphIndex FtGlyphRenderer::GetGlyphIndexVariant(const FtFont &font, Utf32 unicode, UInt32 variantSelector) const
            {
                FEATSTD_UNUSED(variantSelector);
                return GetGlyphIndex(font, unicode, false);
            }

            // ----------------------------------------------------------------------------
            bool FtGlyphRenderer::GetGlyphBitmap(GlyphBitmap &bmp, const FtFont &font, GlyphIndex glyph) const
            {
                if (m_renderInfo.useCache) {
                    return LookupCacheData(bmp, font, glyph);
                }
                else{
                    return LoadImageData(bmp, font, glyph);
                }
            }
            // ----------------------------------------------------------------------------
            PixelPosition2D FtGlyphRenderer::GetGlyphKerning(GlyphIndex currentGlyph, GlyphIndex nextGlyph, const FtFont& font) const
            {
                PixelPosition2D advance = {0, 0};
                if (font.m_properties.m_hasKerning == 1) {
                    FT_Face face = GetFace(font);
                    if (face != 0) {
                        FT_Vector delta;
                        if (FT_Get_Kerning(face, currentGlyph, nextGlyph, FT_UInt(FT_KERNING_DEFAULT), &delta) == 0) {
                            advance.x = static_cast<PixelPosition>(delta.x / 64);
                            advance.y = static_cast<PixelPosition>(delta.y / 64);
                        }
                    }
                }
                return advance;
            }

            // ----------------------------------------------------------------------------
            FT_Face FtGlyphRenderer::GetFace(const FtFont &font) const
            {
                FTC_ScalerRec   scaler;

                scaler.face_id = font.m_type.face_id;
                scaler.width = static_cast<FT_UInt>(font.m_type.width);
                scaler.height  = static_cast<FT_UInt>(font.m_type.height);
                scaler.pixel   = 1;
                scaler.x_res   = 0;
                scaler.y_res   = 0;

                //lookup size instead of face, because face lookup does not insure creation of size.
                FT_Size size;
                FT_Error error = FTC_Manager_LookupSize(FtFontEngine::CacheMgr(), &scaler, &size );
                if ( error == 0 ) {
                    return size->face;
                }
                else {
                    return 0;
                }
            }

            // ----------------------------------------------------------------------------
            bool FtGlyphRenderer::LookupCacheData(GlyphBitmap &bmp, const FtFont &font, GlyphIndex glyph) const
            {
#ifndef FT_USE_DEFAULT_GLYPH_CACHE
                FtCgcItem item;
                FtCgcAccess cacheAccess;
                if (m_renderInfo.access != 0) {
                    cacheAccess = m_renderInfo.access->GetAccess();
                }
                else if (m_renderInfo.renderGlyph) {
                    cacheAccess = FtFontEngine::CgcDefaultAccessor();
                }
                else {
                    cacheAccess = 0;
                }

                bool ok = FtCustomGlyphCache_Lookup(FtFontEngine::CgcCache(),
                                                    &font.m_type,
                                                    cacheAccess,
                                                    glyph,
                                                    &item,
                                                    0) == 0;
#else // ndef FT_USE_DEFAULT_GLYPH_CACHE
                FTC_SBit item;
                bool ok = FTC_SBitCache_Lookup(FtFontEngine::SBitCache(),
                                               &font.m_type,
                                               charIndex,
                                               &item,
                                               0) == 0;
#endif //FT_USE_DEFAULT_GLYPH_CACHE
                if (ok) {
                    bmp.pixels = reinterpret_cast<GlyphBitmap::PixelBufferType>(item->buffer);

                    bmp.left = static_cast<GlyphBitmap::PositionType>(item->left);
                    bmp.top = static_cast<GlyphBitmap::PositionType>(item->top);
                    bmp.width = static_cast<GlyphBitmap::SizeType>(item->width);
                    bmp.height = static_cast<GlyphBitmap::SizeType>(item->height);

                    bmp.xadvance = static_cast<GlyphBitmap::AdvanceType>(item->xadvance);
                    bmp.yadvance = static_cast<GlyphBitmap::AdvanceType>(item->yadvance);
                    bmp.pitch = static_cast<GlyphBitmap::PitchType>(item->pitch);
                    bmp.format = FtPixelModeToFormat(static_cast<FT_Pixel_Mode>(item->pixelmode));

                    return true;
                }
                else {
                    return false;
                }
            }

            // ----------------------------------------------------------------------------
            bool FtGlyphRenderer::LoadImageData(GlyphBitmap &bmp, const FtFont& font, GlyphIndex glyph) const
            {
                FTC_ScalerRec   scaler;
                FT_Size         size;

                scaler.face_id = font.m_type.face_id;
                scaler.width = static_cast<FT_UInt>(font.m_type.width);
                scaler.height  = static_cast<FT_UInt>(font.m_type.height);
                scaler.pixel   = 1;
                scaler.x_res   = 0;
                scaler.y_res   = 0;

                //create bitmap
                FT_Error error = FTC_Manager_LookupSize(FtFontEngine::CacheMgr(), &scaler, &size );
                if ( error == 0 )
                {
                    FT_Face face = size->face;
                    FT_Int32 flags = font.m_type.flags.loadFlags;

                    error = FT_Load_Glyph( face, glyph, flags );
                    if ( error == 0 )
                    {
                        FT_GlyphSlot  slot   = face->glyph;
                        FT_Render_Mode renderFlags = static_cast<FT_Render_Mode>(font.m_type.flags.renderFlags);
                        error = FT_Render_Glyph( slot, renderFlags );
                        if ( error == 0 )
                        {
                            FT_Bitmap*    bitmap = &slot->bitmap;
                            FT_Pos        xadvance;
                            FT_Pos        yadvance;

                            /* approximate advance to closest pixel */
                            xadvance = (slot->advance.x + 32) / 64;
                            yadvance = (slot->advance.y + 32) / 64;

                            if (bitmap->pixel_mode == FT_PIXEL_MODE_MONO)
                            {
                                if (!FtGlyphRenderer::ConvertMonochromToGray(bitmap))
                                {
                                    return false;
                                }
                            }
                            //TODO: check overflow

                            bmp.pixels = reinterpret_cast<GlyphBitmap::PixelBufferType>(bitmap->buffer);

                            bmp.left = static_cast<GlyphBitmap::PositionType>(slot->bitmap_left);
                            bmp.top = static_cast<GlyphBitmap::PositionType>(slot->bitmap_top);
                            bmp.width = static_cast<GlyphBitmap::SizeType>(bitmap->width);
                            bmp.height = static_cast<GlyphBitmap::SizeType>(bitmap->rows);

                            bmp.xadvance = static_cast<GlyphBitmap::AdvanceType>(xadvance);
                            bmp.yadvance = static_cast<GlyphBitmap::AdvanceType>(yadvance);

                            bmp.pitch = static_cast<GlyphBitmap::PitchType>(bitmap->pitch);
                            bmp.format = FtPixelModeToFormat(static_cast<FT_Pixel_Mode>(bitmap->pixel_mode));
                        }
                    }
                }

                return error == 0;
            }

            // ----------------------------------------------------------------------------
            bool FtGlyphRenderer::LoadContourPoint(TextCoordinate &contour, const FtFont& font, GlyphIndex glyph, UInt32 pointIndex) const
            {
                FTC_ScalerRec   scaler;
                FT_Size         size;

                scaler.face_id = font.m_type.face_id;
                scaler.width = static_cast<FT_UInt>(font.m_type.width);
                scaler.height  = static_cast<FT_UInt>(font.m_type.height);
                scaler.pixel   = 1;
                scaler.x_res   = 0;
                scaler.y_res   = 0;

                //create bitmap
                FT_Error error = FTC_Manager_LookupSize(FtFontEngine::CacheMgr(), &scaler, &size );
                if ( error == 0 )
                {
                    FT_Face face = size->face;
                    FT_Int32 flags = FT_LOAD_DEFAULT;

                    error = FT_Load_Glyph( face, glyph, flags );
                    if ( error == 0 )
                    {
                        FT_GlyphSlot  slot   = face->glyph;
                        if ((slot->outline.n_points < 0) || (pointIndex >= static_cast<UInt32>(slot->outline.n_points))) {
                            //index out of bounds.
                            return false;
                        }
                        contour.SetX(static_cast<PixelPosition>(slot->outline.points[pointIndex].x / 64));
                        contour.SetY(static_cast<PixelPosition>(slot->outline.points[pointIndex].y / 64));
                    }
                }

                return error == 0;
            }

            // Conversion has been adapted from FT_Bitmap_Convert function in freetype library 
            // in conjunction with the described patch on 
            // https://lists.nongnu.org/archive/html/freetype-devel/2010-03/msg00054.html
            bool FtGlyphRenderer::ConvertMonochromToGray(FT_Bitmap *source)
            {
                if (0 == source)
                {
                    return false;
                }
                if (source->pixel_mode != FT_PIXEL_MODE_MONO)
                {
                    return false;
                }
                FT_Bitmap *target = TEXTENGINE_TRANSIENT_NEW(FT_Bitmap);
                if (0 == target)
                {
                    return false;
                }

                FT_Bitmap_New(target);
                FT_Byte*  s;
                FT_Byte*  t;
                FT_Int target_pitch;

                target->pixel_mode = FT_PIXEL_MODE_GRAY;
                target->rows = source->rows;
                target->width = source->width;
                target_pitch = static_cast<FT_Int>(source->width);

                if ((target_pitch > 0) &&
                    (static_cast<FT_ULong>(target->rows) > FT_ULONG_MAX / static_cast<FT_ULong>(target_pitch)))
                {
                    TEXTENGINE_DELETE(target);
                    return false;
                }

                target->buffer = TEXTENGINE_TRANSIENT_NEW_ARRAY(FT_Byte, target->rows * static_cast<FT_UInt>(target_pitch));
                if (0 == target->buffer)
                {
                    TEXTENGINE_DELETE(target);
                    return false;
                }

                target->pitch = target->pitch < 0 ? -target_pitch : target_pitch;
                s = source->buffer;
                t = target->buffer;

                /* take care of bitmap flow */
                if (source->pitch < 0)
                {
                    s -= source->pitch * static_cast<FT_Int>(source->rows - 1);
                }
                if (target->pitch < 0)
                {
                    t -= target->pitch * static_cast<FT_Int>(target->rows - 1);
                }

                FT_UInt  i;

                target->num_grays = 2;

                for (i = source->rows; i > 0; i--)
                {
                    FT_Byte*  ss = s;
                    FT_Byte*  tt = t;
                    FT_UInt   j;

                    /* get the full bytes */
                    for (j = source->width >> 3; j > 0; j--)
                    {
                        FT_UInt  val = ss[0]; /* avoid a byte->int cast on each line */

                        tt[0] = (val & 0x80U) != 0 ? 0xFF : 0;
                        tt[1] = (val & 0x40U) != 0 ? 0xFF : 0;
                        tt[2] = (val & 0x20U) != 0 ? 0xFF : 0;
                        tt[3] = (val & 0x10U) != 0 ? 0xFF : 0;
                        tt[4] = (val & 0x08U) != 0 ? 0xFF : 0;
                        tt[5] = (val & 0x04U) != 0 ? 0xFF : 0;
                        tt[6] = (val & 0x02U) != 0 ? 0xFF : 0;
                        tt[7] = (val & 0x01U) != 0 ? 0xFF : 0;

                        tt += 8;
                        ss += 1;
                    }

                    /* get remaining pixels (if any) */
                    j = source->width & 7U;
                    if (j > 0)
                    {
                        FT_UInt  val = *ss;

                        for (; j > 0; j--)
                        {
                            tt[0] = (val & 0x80U) != 0 ? 0xFF : 0;
                            val <<= 1;
                            tt += 1;
                        }
                    }

                    s += source->pitch;
                    t += target->pitch;
                }

                TEXTENGINE_DELETE_ARRAY(source->buffer);
                *source = *target;
                TEXTENGINE_DELETE(target);

                return true;
            }
        }    // namespace Internal
    }    // namespace TextRendering
}    // namespace Candera
