//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_FtCustomGlyphCache_h)
#define Candera_FtCustomGlyphCache_h

#include <Candera/TextEngine/Freetype/FtInclude.h>

#ifdef __cplusplus
namespace Candera {
    namespace TextRendering {
        namespace Internal {
#endif

FT_BEGIN_HEADER

/*************************************************************************/
/* <Struct>                                                              */
/*    FtCgcItem                                                          */
/*                                                                       */
/* <Description>                                                         */
/*    A very compact structure used to describe a small glyph bitmap.    */
/*                                                                       */
/* <Fields>                                                              */
/*    base       :: The base description of the bitmap.                  */
/*                                                                       */
typedef struct FtCgcItemRec_ {
    FT_Short pixelmode;

    FT_UShort width;
    FT_UShort height;
    FT_Short  left;
    FT_Short  top;

    FT_Short  pitch;

    FT_Short  xadvance;
    FT_Short  yadvance;

    FT_Byte*  buffer;
} FtCgcItemRec;

typedef FtCgcItemRec* FtCgcItem;

typedef struct FtCgcFlags_ {
    FT_UInt32 loadFlags : 21;
    FT_UInt32 renderFlags : 4;
} FtCgcFlags;

typedef struct FtCgcFontTypeRec_ {
    FTC_FaceID  face_id;
    FT_Int      height;
    FT_Int      width;
    FtCgcFlags  flags;
} FtCgcFontTypeRec;

typedef FtCgcFontTypeRec* FtCgcFontType;
typedef const FtCgcFontTypeRec* FtCgcFontTypeConst;


/*************************************************************************/
/* <Struct>                                                              */
/*    FtCustomGlyphCache                                                 */
/*                                                                       */
/* <Description>                                                         */
/*    A small glyph bitmap cache, useful when using custom bitmap        */
/*    storing mechanisms.                                                */
/*                                                                       */
typedef struct FtCustomGlyphCacheRec_*  FtCustomGlyphCache;


/*************************************************************************/
/* <Struct>                                                              */
/*    FtCgcAccess                                                        */
/*                                                                       */
/* <Description>                                                         */
/*    A structure used to provide control to the user of the cache over  */
/*    the way the glyph bitmaps are stored.                              */
/*                                                                       */
/* <Fields>                                                              */
/*    process    :: This is a user function that provides a handle for   */
/*                  a bitmap. It is called by the caching system, when   */
/*                  generating the cache item.                           */
/*                                                                       */
/*    free       :: This is a user function used to destory the user     */
/*                  data associated with a bitmap by means of the handle */
/*                  generated by @process. A whole item is provided, for */
/*                  access to additional information.                    */
/*                                                                       */
/*    check      :: This is a user function used to check if a cached    */
/*                  item is still valid or needs to be reprocessed to    */
/*                  render it                                            */
/*                                                                       */
/*    done       :: This function is called on the access object when    */
/*                  the reference counter reaches 0.                     */
/*                                                                       */
/*    refCount   :: This should hold the reference count of this object. */
/*                                                                       */
typedef struct FtCgcAccessRec_* FtCgcAccess;

typedef FT_Byte*
(*FtCgcAccessProcessBitmap)( FtCgcAccess      access,
                             FT_Bitmap* bitmap,
                             FtCgcItem        cachedItem,
                             FT_UInt glyphIndex,
                             FTC_FaceID faceIdentifier,
                             FT_Int fontHeight,
                             FT_Int fontWidth);
typedef void
(*FtCgcAccessFreeBitmap)( FtCgcAccess      access,
                          FtCgcItem        item );
typedef FT_UInt
(*FtCgcAccessCheckItem)( FtCgcAccess      access,
                         FtCgcItem        item );
typedef void
(*FtCgcAccessDone)( FtCgcAccess   access );

typedef struct FtCgcAccessRec_ {
    FtCgcAccessProcessBitmap processBitmap;
    FtCgcAccessFreeBitmap    freeBitmap;
    FtCgcAccessCheckItem     checkItem;
    FtCgcAccessDone          doneAccess;
    
    FT_UInt                  refCount;
} FtCgcAccessRec;

/*************************************************************************/
/*                                                                       */
/* <Function>                                                            */
/*    FtCustomGlyphCache_New                                             */
/*                                                                       */
/* <Description>                                                         */
/*    Create a new custom cache to store small glyph bitmaps.            */
/*                                                                       */
/* <Input>                                                               */
/*    manager :: A handle to the source cache manager.                   */
/*                                                                       */
/* <Output>                                                              */
/*    acache  :: A handle to the new cache.  NULL in case of error.      */
/*                                                                       */
/* <Return>                                                              */
/*    FreeType error code.  0~means success.                             */
/*                                                                       */
FT_EXPORT( FT_Error )
FtCustomGlyphCache_New(FTC_Manager manager, FtCustomGlyphCache* cache);

/*************************************************************************/
/*                                                                       */
/* <Function>                                                            */
/*    FtCustomGlyphCache_Lookup                                          */
/*                                                                       */
/* <Description>                                                         */
/*    Look up a given small glyph bitmap in a given sbit cache and       */
/*    `lock' it to prevent its flushing from the cache until needed.     */
/*                                                                       */
/* <Input>                                                               */
/*    cache  :: A handle to the source cache.                            */
/*                                                                       */
/*    type   :: A pointer to the glyph image type descriptor.            */
/*                                                                       */
/*    access :: A pointer to an object used to provide used access to    */
/*              glyph bitmap storing.                                    */
/*                                                                       */
/*    gindex :: The glyph index.                                         */
/*                                                                       */
/* <Output>                                                              */
/*    item   :: A handle to a bitmap descriptor.                         */
/*                                                                       */
/*    anode  :: Used to return the address of of the corresponding cache */
/*              node after incrementing its reference count (see note    */
/*              below).                                                  */
/*                                                                       */
/* <Return>                                                              */
/*    FreeType error code.  0~means success.                             */
/*                                                                       */
/* <Note>                                                                */
/*    The bitmap descriptor and its bit buffer are owned by the          */
/*    cache and should never be freed by the application.  They might    */
/*    as well disappear from memory on the next cache lookup, so don't   */
/*    treat them as persistent data.                                     */
/*                                                                       */
/*    If `anode' is _not_ NULL, it receives the address of the cache     */
/*    node containing the bitmap, after increasing its reference count.  */
/*    This ensures that the node (as well as the image) will always be   */
/*    kept in the cache until you call @FTC_Node_Unref to `release' it.  */
/*                                                                       */
/*    If `anode' is NULL, the cache node is left unchanged, which means  */
/*    that the bitmap could be flushed out of the cache on the next      */
/*    call to one of the caching sub-system APIs.  Don't assume that it  */
/*    is persistent!                                                     */
/*                                                                       */
FT_EXPORT( FT_Error )
FtCustomGlyphCache_Lookup( FtCustomGlyphCache   cache,
                           FtCgcFontTypeConst   type,
                           FtCgcAccess          access,
                           FT_UInt              gindex,
                           FtCgcItem            *item,
                           FTC_Node             *anode );

FT_END_HEADER

#ifdef __cplusplus
        }   // namespace Internal
    }   // namespace TextRendering
}   // namespace Candera
#endif

#endif    // Candera_FtCustomGlyphCache_h

