//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_Font_h)
#define Candera_Font_h

#include <Candera/TextEngine/FontEngineDefinition.h>
#include <Candera/TextEngine/TextEngineStringBufferAppenders.h>
#include <Candera/System/Container/ForwardIterator.h>

#include <Candera/TextEngine/FontSize.h>
#include <Candera/TextEngine/FontStore.h>
#include <Candera/TextEngine/Internal/FontStyleIterator.h>
#include <Candera/TextEngine/FontEngine.h>


#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSection.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif

namespace Candera {

    namespace TextRendering {

        /** @addtogroup CanderaTextEngine
         *  @{
         */

        /**
         *  @brief Defines a specific font face and size
         */
        class Font : public FontEngineDefinition::Font {
            FEATSTD_TYPEDEF_BASE(FontEngineDefinition::Font);

        public:
            FEATSTD_LINT_CURRENT_SCOPE(1511, "symbol hidden from public API with using, hiding is accepted")
            typedef ForwardIterator<Internal::FontStyleIterator> FontStyleIterator;

            /**
             *  Constructor
             */
            Font();

            /**
             * Destructor
             */
            virtual ~Font();

            /**
            * prepare the font object for text rendering with the given face and size
            * @param faceName name of the font face (e.g. Arial). This string can be released as soon
            *                 as the function returns.
            * @param Height font height in pixel
            * @param faceIndex index of the face
            * @return true if setup succeeded, false otherwise
            * @see Font::Setup
            */
            bool Setup(const Char *faceName, Int16 Height, UInt8 faceIndex = 0)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(GetLock());
#endif
                FontSize fontSize(Height);
                FontStore* store = FontEngine::Instance().GetFontStore();
                if (0 != store)
                {
                    m_fontBuffer = store->GetFontBuffer(faceName, faceIndex);
                }
                return Base::Setup(faceName, fontSize, faceIndex);
            }

            /**
             *  prepare the font object for text rendering with the given face and size
             *  @param faceName name of the font face (e.g. Arial). This string can be released as soon
             *                  as the function returns.
             *  @param fontSize font height/width representation
             *  @param faceIndex font face index (if multipe faces are present). Limitted to 256 faces per font.
             *  @return true if setup succeeded, false otherwise
             */
            bool Setup(const Char *faceName, FontSize const& fontSize, UInt8 faceIndex = 0)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(GetLock());
#endif
                FontStore* store = FontEngine::Instance().GetFontStore();
                if (0 != store)
                {
                    m_fontBuffer = store->GetFontBuffer(faceName, faceIndex);
                }
                return Base::Setup(faceName, fontSize, faceIndex);
            }

            /**
             *  Retrieve metrics of the font object
             *  @param metrics the metrics data structure to receive the metrics data
             *  @return true if metrics could be retrieved, false otherwise
             */
            bool GetMetrics(TextRendering::Metrics &metrics) const
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(GetLock());
#endif
                return Base::GetMetrics(metrics);
            }

            /**
            * Returns the actual font height in pixel.
            * Note: scale factors are already included if set.
            * To get plain sizes, request GetFontSize().
            * @return font height in pixel.
            * @see Font::GetHeight
            */
            Int16 GetHeight() const {
                return Base::GetHeight();
            }

            /**
            * Returns the actual font width in pixel.
            * Note: scale factors are already included if set.
            * To get plain sizes, request GetFontSize().
            * @return font width in pixel.
            * @see Font::GetWidth
            */
            Int16 GetWidth() const {
                return Base::GetWidth();
            }

            /**
            *  Set font height in pixel
            *  For compatibility reasons: Setting the height also sets the width/scale of the font.
            *  If a specific width/scale is requested, use SetFontSize instead.
            *  @param height font height
            */
            void SetHeight(Int16 height)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(GetLock());
#endif
                Base::SetHeight(height);
            }

            /**
            *  Returns the font height in pixel.
            *  @return font height in pixel.
            */
            FontSize GetFontSize() const {
                return Base::GetFontSize();
            }

            /**
            *  Set font size in pixel
            *  It sets the whole fontSize properties.
            *  Standard font behavior is width == height.
            *  @param fontSize font size properties
            */
            void SetFontSize(FontSize const& fontSize)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(GetLock());
#endif
                Base::SetFontSize(fontSize);
            }

            /**
             *  Returns the name of the font face.
             *  @return the name of the font face.
             */
            const Char* GetFaceName() const {
                return Base::GetFaceName();
            }

            /**
             * Returns the family name of the current font face.
             * Availability of the Char* is granted as long as this Font exists and
             * this method has not been called again.
             * @return the family name of the font face.
             */
            const Char* GetFamilyName() const {
                return Base::GetFamilyName();
            }

            /**
             * Returns the style name of the current font face.
             * @return the current style name of the font face.
             */
            const Char* GetStyleName() const {
                return Base::GetStyleName();
            }

            /**
            *  Returns the index of the font face.
            *  @return the index of the font face.
            */
            UInt8 GetFaceIndex() const {
                return Base::GetFaceIndex();
            }

            /**
            *  Returns the id of the font face.
            *  @return The id of the font face.
            */
            UInt8 GetFaceId() const {
                return Base::GetFaceId();
            }

            /**
            * Return offset of the font baseline.
            * @return offset of the font baseline.
            */
            Int16 GetBaseLineOffset() const {
                return Base::GetBaseLineOffset();
            }

            /**
            * Return whether the font is valid or not.
            * @return true if the font is valid, false otherwise.
            */
            bool IsValid() const {
                return (Base::GetFaceId() != FontStore::InvalidDescriptorId);
            }

            /**
             * Configure the output type of the glyph rendering process.
             * This request is not binding. TextRenderContext needs to react the
             * GlyphBitmap::Format, that might be different from this value.
             * See SetHintingMode for restricitons on hinting.
             * Default is GlyphBitmap::Grayscale.
             * @param format Format requested when rendering this font.
             */
            void SetRequestedGlyphFormat(GlyphBitmap::Format format) {
                return Base::SetRequestedGlyphFormat(format);
            }
            /**
             * Retrieve the value set by SetRequestedGlyphFormat.
             * @return Format requested when rendering this font.
             */
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Font::GetRequestedGlyphFormat, CANDERA_LINT_REASON_NONCONSTMETHOD)
            GlyphBitmap::Format GetRequestedGlyphFormat() {
                return Base::GetRequestedGlyphFormat();
            }

            /**
             * Configure the driver autohinter.
             * It is not recommanded to use the monochrome hinter with translucent
             * render mode, and viceversa (i.e. translucent hinter with monochrome
             * render mode).
             * Default is GlyphHinter::Normal.
             * @param mode Hinting mode used for rendering this font.
             */
            void SetHintingMode(GlyphHinter::Mode mode) {
                return Base::SetHintingMode(mode);
            }
            /**
             * Retrieve the value set by SetHintingMode.
             * @return Hinting mode used for rendering this font.
             */
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::TextRendering::Font::GetHintingMode, CANDERA_LINT_REASON_NONCONSTMETHOD)
            GlyphHinter::Mode GetHintingMode() {
                return Base::GetHintingMode();
            }

            /**
             * Enable hinting when loading glyphs. If hinting is disabled, glyphs may appear blurred.
             * Default is true.
             * @param enabled True means enabled hinting, false disabled.
             */
            void SetHintingEnabled(bool enabled) {
                return Base::SetHintingEnabled(enabled);
            }
            /**
             * Retrieve the value set by SetHintingEnabled.
             * @return True means enabled hinting, false disabled.
             */
            bool IsHintingEnabled() {
                return Base::IsHintingEnabled();
            }

            /**
             * Enable vertical layout of glyphs, when supported by font.
             * Default is false.
             * @param enabled True means enabled vertical layout, false disabled.
             */
            void SetVerticalLayoutEnabled(bool enabled) {
                return Base::SetVerticalLayoutEnabled(enabled);
            }
            /**
             * Retrieve the value set by SetVerticalLayoutEnabled.
             * @return True means enabled vertical layout, false disabled.
             */
            bool IsVerticalLayoutEnabled() {
                return Base::IsVerticalLayoutEnabled();
            }

            /**
             * Enable usage of the automatic hinter of the font driver.
             * When disabled either the native hinter of the font is used, 
             * or hinting is diabled altogether.
             * Default is true.
             * @param enabled True means enabled auto-hinting, false disabled.
             */
            void SetAutohintEnabled(bool enabled) {
                return Base::SetAutohintEnabled(enabled);
            }
            /**
             * Retrieve the value set by SetAutohintEnabled.
             * @return True means enabled auto-hinting, false disabled.
             */
            bool IsAutohintEnabled() {
                return Base::IsAutohintEnabled();
            }

            /**
             * Prefer the automatic hinter of the font driver, instead of the native one stored within the font.
             * Default is false.
             * @param enabled True means enabled forced auto-hinting, false disabled.
             */
            void SetForceAutohintEnabled(bool enabled) {
                return Base::SetForceAutohintEnabled(enabled);
            }
            /**
             * Retrieve the value set by SetForceAutohintEnabled.
             * @return True means enabled forced auto-hinting, false disabled.
             */
            bool IsForceAutohintEnabled() {
                return Base::IsForceAutohintEnabled();
            }

            /**
             * Crop bitmaps to size of the non-transparent pixels.
             * Default is false.
             * @param enabled True means enabled cropping, false disabled.
             */
            void SetCropEnabled(bool enabled) {
                return Base::SetCropEnabled(enabled);
            }
            /**
             * Retrieve the value set by SetCropEnabled.
             * @return True means enabled cropping, false disabled.
             */
            bool IsCropEnabled() {
                return Base::IsCropEnabled();
            }

            /**
             * Enable failure upon detecting binary corruption. When disabled,
             * an attempt is made at rendering with the available data.
             * Default is false.
             * @param enabled True means enabled verifications, false disabled.
             */
            void SetPedanticVerificationEnabled(bool enabled) {
                return Base::SetPedanticVerificationEnabled(enabled);
            }
            /**
             * Retrieve the value set by SetPedanticVerificationEnabled.
             * @return True means enabled verifications, false disabled.
             */
            bool IsPedanticVerificationEnabled() {
                return Base::IsPedanticVerificationEnabled();
            }

            /**
             * @param fontFace Name of the font face.
             * @return a ForwardIterator that iterates over all the style names contained by given font face.
             */
            static FontStyleIterator GetFontStyleIterator(const Char* fontFace) {
                return FontStyleIterator(Internal::FontStyleIterator(fontFace), Internal::FontStyleIterator());
            }

            /**
             * Equality comparison operator. Fonts are equal if they reference
             * the same font resource, and are configured with the same paramenters.
             * @param rhs Font agains which equality shall be tested.
             * @return True means equal.
             */
            bool operator == (const Font& rhs) const {
                return Base::operator ==(rhs);
            }

            /**
             * Equality comparison operator. Fonts are equal if they reference
             * the same font resource, and are configured with the same paramenters.
             * @param rhs Font agains which equality shall be tested.
             * @return True means not equal.
             */
            bool operator != (const Font& rhs) const {
                return !operator==(rhs);
            }

            FEATSTD_LINT_CURRENT_SCOPE(1509, "base class has intentionally no virtual dtor")

        private:
#ifdef FEATSTD_THREADSAFETY_ENABLED
            static FeatStd::Internal::CriticalSection * GetLock();
#endif

            friend class Internal::FontStyleIterator;

            static const Char* GetStyleName(const Char* faceName, UInt8 styleIndex)  {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(GetLock());
#endif
                return Base::GetStyleName(faceName, styleIndex);
            }
            FontBuffer::SharedPointer m_fontBuffer; // shared FontBuffer, member solely for reference counting purpose
        };

        /** @} */ // end of CanderaTextEngine

    }// namespace TextRendering

}// namespace Candera


#endif// Candera_Font_h
