//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextValidator_h
#define Candera_TextEngine_TextValidator_h

#include <FeatStd/Container/Vector.h>
#include <FeatStd/Util/ValidationHelperBase.h>
#include <Candera/Candera.h>

namespace Candera {

    class Scene2D;

    namespace TextRendering {

        /** @addtogroup CanderaTextEngine
        *  @{
        */

        /**
        * @brief ITextValidationUser is an interface for a user which will be attached to a TextValidator.
        *
        * TextValidationGroup also needs this interface to register a user
        * and will indirectly attach a TextValidator to this user.
        */
        class ITextValidationUser {
        public:
            virtual ~ITextValidationUser() {}
            /**
            * Attaches an actual validation helper to the user
            * When a TextValidationGroup is used, this will be called indirectly by the attacher
            * of TextValidationGroup
            */
            virtual bool AttachValidationHelper(FeatStd::ValidationHelperBase* validationHelper) = 0;

            /**
            * Detaches the validation helper
            * When a TextValidationGroup is used, this will be called indirectly by the detacher
            * of TextValidationGroup
            */
            virtual bool DetachValidationHelper(FeatStd::ValidationHelperBase* validationHelper) = 0;

            /**
            * Indicator whether this user currently has an asynchronous update or not
            * If everything is synchronous this value is true in general
            * --> Synchronous == always valid user, because triggering an update handling it and setting the result
            * is done in one go
            */
            virtual bool IsAsyncPreRenderEnabled()const = 0;

            /**
            * Will be called when:
            * 1. all users of a group (if a group is available)
            * 2. or when the user (if no group is available)
            * are changing state from invalid to valid
            */
            virtual void TriggerValidUpdate() = 0;

            /**
            * Indicator whether this user always wants the trigger valid update despite any cases.
            * @return true is a forced trigger update call
            */
            virtual bool IsTriggerUpdateForced() = 0;
        protected:
            ITextValidationUser() {}
        };


        class TextValidationGroup;
        /**
        * @brief TextValidator is the actual Validator of an 'ITextValidationUser'.
        *
        * It works as follows:
        * Text - nothing changed: --> IsSingleValid() == true;
        * IsValid() depends on all Users in the group (All users have to be IsSingleValid() == true).
        * Text - trigger update (SetInvalid()): --> IsSingleValid() == false;
        * Text - is updated but results are not set yet: (SetValid()): --> IsSingleValid() == true;
        * the change from invalid to valid triggers a notification on all users (if whole group is valid)
        * internal state has 1 pending result
        * Text - pending data has been handled --> acknowledge with ConfirmValidHandled(): --> IsSingleValid() == true;
        * It is possible to generate a few pending updates without acknowledgment
        */
        class TextValidator : public FeatStd::ValidationHelperBase {
        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(TextValidator);
            TextValidator(ITextValidationUser* toBeValidated, TextValidationGroup * attachedValidationGroup);

            virtual ITextValidationUser* GetValidationUser();

            /**
            * Checks whether the text is valid
            * For full check: Text is valid and the textgroup is also valid (So all texts in group are synced)
            * Use IsValid()
            */
            virtual bool IsSingleValid() const;

            /**
            * Checks whether the text has a pending valid.
            * so the user is valid but a result still has to be confirmed
            */
            virtual bool IsPendingValid() const;

            /**
            * validation of a changed text property. Text would be corrected now (Pending validation acceptance).
            * A valid marker is added.
            * Needs to be overridden otherwise deadlock - because SetValid() calls Group method
            * so it locks Text and then group
            * And all the other functionalities lock group first and afterwards Text
            */
            virtual void SetValid() override;
            /**
            * Checks whether the text and the bound validation group is valid
            * Needs to be overridden otherwise deadlock - because SetValid() calls Group method
            * so it locks Text and then group
            * And all the other functionalities lock group first and afterwards Text
            */
            virtual bool IsValid() const override;
        protected:
            /**
            *   marks that text (properties) has changed - wait for validation
            */
            virtual void SetInvalidImplementation() override;

            /**
            * validation of a changed text property. Text would be corrected now (Pending validation acceptance).
            * A valid marker is added
            */
            virtual void SetValidImplementation() override;

            /**
            * Resets marker for valid and invalid states of a text
            */
            virtual void ResetImplementation() override;

            /**
            * Checks whether the text and the bound validation group is valid
            */
            virtual bool IsValidImplementation() const override;

            /**
            * removes a valid marker: A valid text were properly handled. So the information (marker)
            * which said that there is still a pending valid result can be removed
            */
            virtual void ConfirmValidHandledImplementation() override;

            ITextValidationUser* m_valUser;
            TextValidationGroup* m_validationGroup;
            UInt m_invalidCount : 4;
            UInt m_validCount : 4;


        };

        /** @} */ // end of CanderaTextEngine
    }// namespace TextRendering
}// namespace Candera
#endif // Candera_TextEngine_TextValidator_h
