//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextValidationGroup_h
#define Candera_TextEngine_TextValidationGroup_h

#include <Candera/Environment.h>
#include <FeatStd/Container/Vector.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/TextEngine/Async/TextValidator.h>
#include <FeatStd/Util/ValidationHelperBase.h>
#include <Candera/TextEngine/Async/TextValidationGroupConnector.h>

namespace Candera {
    namespace TextRendering {

        /** @addtogroup CanderaTextEngine
        *  @{
        */

        /**
        * TextValidationGroup represents a group for validation users
        * It is valid when all users in the group are valid.
        * Changing from invalid to valid (complete group has to be valid) triggers 'TriggerValidUpdate' on all
        * users which had changes (invalid to valid).
        * Not triggered are users with: IsAsyncPreRenderEnabled() == false,
        * and valid nodes which never became invalid
        */
        class TextValidationGroup : public FeatStd::ValidationHelperBase {
        public:

            /**
            * Standard ValidationGroup
            * Represents a standard validation group so that there is always a validator
            * to which everyone can bind. (e.g. all TextNode2Ds are bound to this validator per default
            * to sync all TextNode2Ds per default)
            */
            static TextValidationGroup& GetDefaultValidator();

            TextValidationGroup();

            virtual ~TextValidationGroup();

            /**
            * Binds a user to this validation group
            * It creates an own TextValidator for this user and attaches this TextValidator to the user.
            * Additionally it is now bound to the group, so as soon as (but only) when all users in the group
            * are valid, an update will be triggered.
            * NOTE: Previously attached validation users have to be detached from their validator first.
            */
            bool AttachValidationUser(ITextValidationUser * user);

            /**
            * Unbinds a user from this validation group.
            */
            bool DetachValidationUser(ITextValidationUser * user);

            /**
            * Adds a group connector to this group.
            */
            bool AttachGroupConnector(TextValidationGroupConnector::SharedPointer& connector);

            /**
            * Removes a group connector to this group.
            */
            bool DetachGroupConnector(TextValidationGroupConnector::SharedPointer& connector);

            /**
            * Checks whether all users in this group are valid, and triggers ValidationUpdate() on all users
            * This is normally called by the validator of the bound users.
            */
            void HandleValidation();

        protected:

            FeatStd::Internal::Vector<TextValidator::SharedPointer> m_boundUser;
            FeatStd::Internal::Vector<TextValidationGroupConnector::SharedPointer> m_boundConnector;

            /**
            * This function should not be used and therefore hits a breakpoint.
            * The group does not actively need a confirmation.
            */
            virtual void SetInvalidImplementation() override;

            /**
            * This function should not be used and therefore hits a breakpoint.
            * The group does not actively need a confirmation.
            */
            virtual void SetValidImplementation() override;

            /**
            * This function should not be used and therefore hits a breakpoint.
            * The group does not actively need a confirmation.
            * Every validation user should decide independently if it can reset or not.
            * Reset all users could be helpful but can easily produce undesired behavior.
            */
            virtual void ResetImplementation() override;

            /**
            * Checks whether all validation users are valid
            */
            virtual bool IsValidImplementation() const override;

            /**
            * Checks whether a validation user has a pending validation state
            * and trigger their update method.
            */
            virtual void TriggerTextUpdates();

            /**
            * This function should not be used and therefore hits a breakpoint.
            * The group does not actively need a confirmation.
            */
            virtual void ConfirmValidHandledImplementation() override;

        };
        /** @} */ // end of CanderaTextEngine
    }// namespace TextRendering
}// namespace Candera

#endif

