//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextRenderHandleController_h
#define Candera_TextEngine_TextRenderHandleController_h

#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <FeatStd/Async/AsyncRequest.h>
#include <FeatStd/Util/ValidationHelperBase.h>

#include <Candera/TextEngine/Async/TextRenderArgs.h>
#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/Async/TextValidator.h>

CANDERA_UNIT_TEST_CLASS_DECLARATION(AsyncSimulator)
namespace Candera {
    namespace TextRendering {
        namespace Internal {
            /** @addtogroup CanderaTextEngine
            *  @{
            */
            namespace TextRenderState {
                enum Enum /*: UInt8 - cannot be used before c++11*/
                {
                    Free = 0,       ///< Slot is idle (a free slot available to start a new process).
                    Preparing,      ///< Slot has been taken but the arguments have to be set (taken but not queued yet).
                    Waiting,        ///< Slot is queued and waits for execution.
                    Running,        ///< Slot is queued and currently executed.
                    Completed,      ///< Slot has finished calculating - wait for a receptor of the results.
                    PostAnalysis    ///< Slot is finished and the results are currently retrieved. After retrieval Slot will be free again.
                };
            }
            /**
            *  @brief TextRenderHandleController manages the render arguments, render slots and their states.
            *  There are 2 available render slots.
            *
            *  The basic idea is:
            *  - GetFreeArgs(...) - returns a free slot and the arguments (now transforms to preparing slot)
            *    --> set arguments here
            *    --> create async task here
            *  - SetHandle(...) - set the async handle of the preparing slot (now transforms to waiting/running slot)
            *    --> Do something else (wait until a task finished)
            *  - GetRenderCompletedIndex(...) - get a completed slot. The operation on it is done (now transforms to PostAnalysis slot)
            *    --> retrieve the results
            *  - FreeIndex(...) - frees the PostAnalysis slot
            */
            class TextRenderHandleController {
            public:
                FEATSTD_TYPEDEF_SHARED_POINTER(TextRenderHandleController);
                typedef FeatStd::MemoryManagement::SharedPointer<FeatStd::AsyncRequest<bool> > THandleBase;
                typedef TextRenderArgs::SharedPointer TArgsSharedPointer;

                TextRenderHandleController(ITextValidationUser * owner = 0);

                virtual ~TextRenderHandleController();

                FEATSTD_SHARED_POINTER_CREATE_DECLARATION()
                {
                    return TextRenderHandleController::SharedPointer(TEXTENGINE_TRANSIENT_NEW(TextRenderHandleController)());
                }

                /**
                * sets an async process handle (associated to idx)
                * @param idx is the handle index inside the controller
                * @param handle is the actual threading handle for this slot (idx)
                */
                void SetHandle(UInt8 const idx, THandleBase handle);

                /**
                * Set an idx to be finished directly. so basically a synchronous usage does not need to set a handle and the
                * steps Waiting/Running are not needed
                */
                void SetFinishedHandle(UInt8 const idx);

                /**
                * @return idx of a finished process
                * @return in/out-arguments of rendering
                * @param[out] renderResult (return value of text engine)
                */
                UInt8 GetRenderCompletedIndex(TArgsSharedPointer& result, bool& renderResult);

                /**
                * Frees an index (slot) so it can be used again
                * @param idx is the identifier for the slot which will be freed
                */
                void FreeIndex(UInt8 idx);

                /**
                * @return an idx which is invalid and always returned if there is no slot available for a given operation
                */
                UInt8 InvalidIndex() const;

                /**
                * @return the amount of slots which are not free
                */
                UInt8 GetQueuedCount() const;

                /**
                * @return a free slot idx and its arguments
                */
                virtual UInt8 GetFreeArgs(TextRenderArgs::SharedPointer& resArgs);

                /**
                * @return the amount of slots which are in the state 'Completed'
                */
                virtual UInt8 CheckForFinished();

                /**
                * Tries to abort all tasks.
                * A failed abort is a task which is already running
                */
                void AbortAll();

                /**
                * @return the validation helper bound to the user of this controller. It is mainly needed for e.g. abort case in which a previous invalid call has to be validated with
                * a valid call.
                */
                FeatStd::ValidationHelperBase * GetValidationHelper() const { return m_validationHelper; }

                /**
                * Sets the validation helper bound to the user of this controller. It is mainly needed for e.g. abort case in which a previous invalid call has to be validated with
                * a valid call.
                */
                void SetValidationHelper(FeatStd::ValidationHelperBase * val) { m_validationHelper = val; }

                /**
                * @return the validation helper user.
                */
                Candera::TextRendering::ITextValidationUser * GetOwner() const { return m_owner; }

                /**
                * Sets the validation helper user.
                * @param val is the user of this controller
                */
                void SetOwner(Candera::TextRendering::ITextValidationUser * val) { m_owner = val; }
            private:
                FEATSTD_MAKE_CLASS_UNCOPYABLE(TextRenderHandleController);
                FEATSTD_SHARED_POINTER_DECLARATION();

                class Data {
                public:
                    THandleBase m_handle;
                    TArgsSharedPointer m_args;
                    TextRenderState::Enum m_state;
                    bool m_throwAway;
                    Data() : m_handle(), m_args(), m_state(TextRenderState::Free), m_throwAway(false) {}

                    void MoveFrom(Data& data)
                    {
                        if(&data != this) {
                            FEATSTD_DEBUG_ASSERT(m_handle.PointsToNull() && m_state == TextRenderState::Free && m_args.PointsToNull());
                            m_handle = data.m_handle;
                            m_args = data.m_args;
                            m_state = data.m_state;
                            data.m_state = TextRenderState::Free;
                            data.m_args.Release();
                            data.m_handle.Release();
                        }
                    }
                private:
                    FEATSTD_MAKE_CLASS_UNCOPYABLE(Data);
                };

                Data m_data[2];

                void ThrowAwayMarkedSlots();

                bool AbortTask(UInt8 idx);

                UInt8 GetFreeArgsIndex();

                virtual void UpdateAllRenderStates();

                UInt8 ShiftDataToNewestPosition(UInt8 index);

                // Function call during Prepare, PostAnalysis step or 2 running states (not possible)
                // 2 waiting should also be impossible - a waiting task should be replaced by the new task
                bool IsFreeIdxIsValidRequest() const;

                void ResetIndex(UInt8 idx);

                TextRenderState::Enum CheckRenderState(UInt8 idx);

                void Print(char * addText);

                FeatStd::ValidationHelperBase * m_validationHelper;
                ITextValidationUser * m_owner;

                CANDERA_UNIT_TEST_CLASS_FRIEND(AsyncSimulator);
            };

            /** @} */ // end of CanderaTextEngine
        }// namespace Internal
    }// namespace TextRendering
}// namespace Candera

#endif // Candera_TextEngine_TextRenderHandleController_h
