//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/TextEngine/Async/TextRenderDispatcher.h>

#include <FeatStd/Async/AsyncRequest.h>
#include <FeatStd/Async/AsyncRequestDispatcher.h>

#include <Candera/TextEngine/Async/TextRenderArgs.h>
#include <Candera/TextEngine/Async/AsyncTextRenderDispatcher.h>
#include <Candera/TextEngine/Async/ThreadingTextRenderDispatcher.h>

namespace Candera {
    namespace TextRendering {

        TextRenderDispatcher::TextRenderSettings::TextRenderSettings():
            m_defaultTextRenderer(0),
            m_asyncLoopCount(1)
        {
#if defined(CANDERA_TEXTENGINE_WORKER_THREAD_ENABLED) && (!((defined(_FEATSTD_OS_INTERNAL_THREAD_NOT_SUPPORTED)) || (!defined(FEATSTD_THREADSAFETY_ENABLED))))
            m_defaultTextRenderer = &ThreadingTextRenderDispatcher::GetInstance();
#else
            m_defaultTextRenderer = &AsyncTextRenderDispatcher::GetInstance();
#endif
        }

        TextRenderDispatcher * TextRenderDispatcher::TextRenderSettings::GetDefaultTextRenderer() const
        {
            return m_defaultTextRenderer;
        }

        void TextRenderDispatcher::TextRenderSettings::SetDefaultTextRenderer(TextRenderDispatcher * val)
        {
            if (val != 0) {
                m_defaultTextRenderer = val;
            }
        }

        void TextRenderDispatcher::TextRenderSettings::SetAsyncLoopCount(UInt8 const val)
        {
            FEATSTD_DEBUG_ASSERT(val > 0);
            if (val > 0) {
                m_asyncLoopCount = val;
            }
        }

        bool TextRenderDispatcher::TextRenderSettings::SetPriority(FeatStd::Internal::ThreadPriority::Enum priority,
            FeatStd::Internal::Thread::SetPriorityCallback callback)
        {
            return m_defaultTextRenderer->SetPriority(priority, callback);
        }

        TextRenderDispatcher::TextRenderSettings& TextRenderDispatcher::TextRenderSettings::GetInstance()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(TextRenderSettings, s_settings);
            return s_settings;
        }

        TextRenderDispatcher::TextRenderSettings& TextRenderDispatcher::GetTextRenderSettings()
        {
            return TextRenderSettings::GetInstance();
        }


#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSection * TextRenderDispatcher::GetCsLock()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, s_cs);
            return &s_cs;
        }
#endif

        TextRenderHandle TextRenderDispatcher::RenderAsync(TRenderProcess::CallbackType callback, TextRenderArgs::SharedPointer& args, UInt32 const priority)
        {
            return RenderAsync(GetTextRenderSettings().GetDefaultTextRenderer(), callback, args, priority);
        }

        Candera::TextRendering::TextRenderHandle TextRenderDispatcher::RenderAsync(Candera::TextRendering::TextRenderDispatcher * dispatcher, TRenderProcess::CallbackType callback, TextRenderArgs::SharedPointer& args, UInt32 const priority)
        {
            FEATSTD_DEBUG_ASSERT(!args.PointsToNull());
            if (args->GetValidationHelper() != 0) {
                args->GetValidationHelper()->SetInvalid();
            }
            TextRenderHandle c(FEATSTD_NEW(TRenderProcess)(callback, args));
            if ((dispatcher != 0) && (dispatcher->GetDispatcher() != 0)) {
                dispatcher->GetDispatcher()->Schedule(c, priority);
            }
            else {
                c->Execute();
            }
            return c;
        }

    }
}
