//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextRenderArgs_h
#define Candera_TextEngine_TextRenderArgs_h

#include <FeatStd/Base.h>
#include <FeatStd/Util/ValidationHelperBase.h>
#include <Candera/Environment.h>

#include <Candera/TextEngine/TextEngineMemoryPool.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/TextSize.h>
#include <Candera/TextEngine/TextProperties.h>
#include <Candera/TextEngine/LayoutingOptions.h>
#include <Candera/TextEngine/TextRenderContexts/TruncationToGlyphIteratorContext.h>
#include <Candera/TextEngine/Style.h>
#include <Candera/TextEngine/TextRenderContexts/GlyphDataContainer.h>
#include <Candera/TextEngine/Internal/TextProcessProperties.h>


namespace Candera {
    // refactor - TextNode2D in Candera Core part; not part of textengine
    class TextNode2D;
    namespace TextRendering {
        /** @addtogroup CanderaTextEngine
        *  @{
        */
        class TextRenderContext;
        class DummyCacheContext;
        /**
        * @brief TextRenderArgs is used to store all information which is required to render a text.
        *
        * The storage has to be used to provide thread safety whilst processing the text.
        * It also provides the TextNode2D with the results of the procedure, so the TextNode2D can
        * retrieve the results when it is thread safe.
        */
        class TextRenderArgs {
        public:

            FEATSTD_TYPEDEF_SHARED_POINTER(TextRenderArgs);
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION()
            {
                return TextRenderArgs::SharedPointer(TEXTENGINE_TRANSIENT_NEW(TextRenderArgs)());
            }

            /**
            * Sets the maximal area the text can use.
            * @param size defines the maximum area.
            */
            void SetTextAreaSize(TextSize const& size)
            {
                m_clientArea = size;
                m_layoutingOptions.SetSize(size);
            }

            /**
            * Initializes the TextRenderArgs when TextNode2D do not have a layouter attached.
            * @param node TextNode2D to extract the default init infos
            */
            bool DefaultInit(TextNode2D& node);

            /**
            * Initializes the TextRenderArgs when TextNode2D has a layouter attached.
            * @param node provides the TextNode2D which shall be rendered
            * @param clientArea provides the maximum dimension set by the layouter
            * @param layoutBase provides layout properties
            * @param truncationContext represents the context to create the truncation glyph iterator
            */
            bool MeasureInit(TextNode2D& node,
                             TextSize const& clientArea,
                             LayoutingOptions layoutBase,
                             TruncationToGlyphIteratorContext::SharedPointer& truncationContext);

            /**
            * Initializes the TextRenderArgs when the TextProcessHelper methods are used
            * @param style provides the font information
            * @param clientArea provides the maximum dimension set by the layouter
            * @param txtProps provides the text
            * @param cacheContext provides the text/glyph cache
            * @param layoutBase provides layout properties
            * @param truncationContext represents the context to create the truncation glyph iterator
            */
            bool MeasureInit(SharedStyle::SharedPointer& style,
                             TextSize const& clientArea, 
                             TextProperties& txtProps, 
                             Internal::TextProcessProperties::TextCache* cacheContext,
                             LayoutingOptions layoutBase, 
                             TruncationToGlyphIteratorContext::SharedPointer& truncationContext);

            /**
            * Sets the validation base to evaluate whether the text is valid or not
            * @param val is the validation base to evaluate whether the text is valid or not 
            */
            void SetValidationHelper(FeatStd::ValidationHelperBase* val) { m_validationHelper = val; }

            /**
            * @return the validation base to evaluate whether the text is valid or not
            */
            FeatStd::ValidationHelperBase* GetValidationHelper() const { return m_validationHelper; }

            /**
            * It sets the results of text processing and validates the text node.
            * This method should be called in the text processing thread.
            * @param layoutRect defines the rectangle set by layout
            * @param textRect defines the rectangle which is required to render the text
            * @return determines whether the results could be set or not
            */
            bool SetResults(TextRect const& layoutRect, TextRect const& textRect);

            /**
            * @return the text style
            */
            inline const SharedStyle::SharedPointer& GetStyle() const
            {
                return m_style;
            }

            /**
            * @return the cache context
            */
            inline const TextRenderContext* GetCacheContext() const
            {
                return m_cacheContext;
            }

            /**
            * @return the maximum area a text can use
            */
            inline const TextSize& GetArea() const
            {
                return m_clientArea;
            }

            /**
            * @return the truncation context which contains the container for truncation replacement glyphs
            */
            inline TruncationToGlyphIteratorContext::SharedPointer GetTruncationContext() const
            {
                return m_truncationContext;
            }

            /**
            * @return the length of the text
            */
            inline const TextPosition& GetTextLength() const
            {
                return m_textLength;
            }

            /**
            * @return the text
            */
            inline const TextProperties& GetText() const
            {
                return m_textProps;
            }

            /**
            * Sets the glyph container of the text in which the glyphs will be stored
            * @param textVector represents the glyph iterator container 
            */
            inline void SetGlyphContainer(GlyphDataContainer::SharedPointer& textVector)
            {
                m_textVector = textVector;
            }

            /**
            * @return the glyph container of the text
            */
            inline GlyphDataContainer::SharedPointer GetGlyphContainer() const
            {
                return m_textVector;
            }

            /**
            * @return a reference to the layout options so they can be changed.
            */
            inline LayoutingOptions& GetLayoutingOptionsRef()
            {
                return m_layoutingOptions;
            }

            /**
            * @return the actual used layout rectangle. Only valid after preprocessing/SetResults
            */
            inline const Candera::TextRendering::TextRect& GetLayoutRectangle() const
            {
                return m_LayoutRectangle;
            }

            /**
            * @return the actual used text rectangle. Only valid after preprocessing/SetResults
            */
            inline const Candera::TextRendering::TextRect& GetTextRectangle() const
            {
                return m_textRectangle;
            }


            virtual ~TextRenderArgs();

        protected:
            FeatStd::ValidationHelperBase* m_validationHelper;
        private:
            LayoutingOptions m_layoutingOptions;
            Candera::TextRendering::TextRect m_LayoutRectangle;
            Candera::TextRendering::TextRect m_textRectangle;
            TextProperties m_textProps;
            TextSize m_clientArea;
            TextPosition m_textLength;
            SharedStyle::SharedPointer m_style;
            TextNode2D* m_textNode;
            const TextRenderContext* m_cacheContext;
            DummyCacheContext * m_dummyCacheContext;
            GlyphDataContainer::SharedPointer m_textVector;
            TruncationToGlyphIteratorContext::SharedPointer m_truncationContext;
            bool m_isAsyncCall;

            void Cleanup();
            TextRenderArgs();
            FEATSTD_SHARED_POINTER_DECLARATION();
            FEATSTD_MAKE_CLASS_UNCOPYABLE(TextRenderArgs);
        };
        /** @} */ // end of CanderaTextEngine
    }// namespace TextRendering
}// namespace Candera
#endif // Candera_TextEngine_TextRenderArgs_h
