//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(Candera_AlignTextMeasureContext_h)
#define Candera_AlignTextMeasureContext_h

#include <Candera/TextEngine/TextMeasureContext.h>
#include <Candera/TextEngine/TextLayoutStrategy.h>
#include <Candera/TextEngine/TextRect.h>

namespace Candera { namespace TextRendering {
    class Style;

/** @addtogroup CanderaTextEngine
 *  @{
 */

/**
 *  @brief AlignTextMeasureContext is used to measure the movement of the
 *  text cursor.
 *
 *  This class builds the smallest axis aligned rectangle that contains the
 *  the whole surface swept by the cursor when moving while processing the text.
 *  The cursor is considered to be a segment of length equal to the style
 *  height.
 *
 *  This class is best used when aligning left-to-right text to the right or
 *  right-to-left text to the left, as the trailing spaces are usually required
 *  to be ignored in alignment.
 *
 *  AlignLayoutStrategy is required to be used in conjunction with AlignTextMeasureContext
 *  for multiline texts.
 */
class AlignTextMeasureContext : public TextMeasureContext {
public:
    /**
     * Constructor.
     * @param style Style from which font metrics are retrieved.
     * @param referenceContext Context to retrieve caching strategy from.
     */
    AlignTextMeasureContext(const Style& style, const TextRenderContext* referenceContext = 0);

    void SignalNewLine() { m_isWhiteSpaceLeading = true; }

    void SetIgnoreTrailingSpaces(bool value) { m_ignoreTrailingSpaces = value; }

    /**
     * Retrieve the rectangle computed by the context.
     * The rectangle is equal to TextRect() if nothing was measured.
     * @return The computed rectangle.
     */
    const TextRect& GetTextRectangle() const;

    // Overrides function from TextRenderContext
    virtual GlyphOrder GetGlyphOrder() const override { return OriginalGlyphOrder; }

private:
    // Overrides function from TextMeasureContext.
    virtual void Measure(PixelPosition x, PixelPosition y, const GlyphBitmap &glyph) ;

    bool m_isWhiteSpaceLeading;
    bool m_ignoreTrailingSpaces;
    PixelPosition m_ascender;
    PixelPosition m_descender;
    TextRect m_textRectangle;
};


/**
*  @brief AlignLayoutStrategy is used signal AlignTextMeasureContext about new lines.
*/
class AlignLayoutStrategy : public TextLayoutStrategy
{
public:
    /**
     * Constructor.
     * @param measureContext Asociated AlignTextMeasureContext that will be notified when new lines occur.
     * @param strategy Additional TextLayoutStrategy that will be attached in the rendering process.
     */
    AlignLayoutStrategy(AlignTextMeasureContext& measureContext, TextLayoutStrategy* strategy);

    /**
     * Signal the begining of a line to the measure context.
     */
    virtual Action OnInit(Pass pass);

    /**
     * Signal the begining of a line to the measure context.
     */
    virtual Action OnLineBreak(const Char* textCursor, LineBreakType type);

    virtual Action OnGlyphProcessing(TextPosition textPosition);

private:
    AlignTextMeasureContext m_alignMeasureContext;
    TextLayoutStrategy* m_next;
};

 /** @} */ // end of CanderaTextEngine

}// namespace TextRendering
}// namespace Candera

#endif// Candera_AlignTextMeasureContext_h
